gdd = require('./models/gdd.js');
gdd_current = require('./models/gdd_current.js');
normals = require('./models/normals.js');

function isNumeric(value) {
    return /^-?\d+$/.test(value);
}

exports.year_gdd = function (req, res) {
    if (!isNumeric(req.params.year)) {
        res.status(400).send({"parameter error": "\"year\" parameter is not an integer"})
    }
    var year = parseInt(req.params.year);
    var product = req.params.product;

    if (typeof req.body.latitude === "undefined" || typeof req.body.longitude === "undefined") {
        res.status(400).send({"parameter error": "missing latitude or longitude"})
    }

    var latitude = parseFloat(req.body.latitude)
    var longitude = parseFloat(req.body.longitude)

    var query = {
        location: { 
            "$near": {
                "$geometry": {
                    "type": "Point", 
                    "coordinates": [longitude, latitude]
                },
            },
        },
        year: year,
    }

    var projection = {
        min_temps: 1,
        max_temps: 1,
        location: 1,
    }
    
    var t_base = 50
    var t_max = 86
    var t_min = 50 

    errors = []

    if (year < 1981 || year > new Date().getFullYear()) {
        errors.push({
            parameter_error: "year",
            message: year.toString() + " is out of bounds for GDD calculations. Must be between 1981 - Current Year"
        });
    }

    if (latitude < 24.083334 || latitude > 49.916668) {
        errors.push({
            parameter_error: "latitude",
            message: latitude.toString() + " is out of bounds for GDD calculations. Must be between 24.083334 - 49.916668"
        });
    }

    if (req.body.hasOwnProperty("t_base")) {
        t_base = parseFloat(req.body.t_base);
        if (t_base < t_min) {
            t_min = t_base;
        }
    } else {
        switch (product) {
            case "soybean":
            case "corn":
            case "sunflower":
            case "tomato":
            case "sugar_beat":
                t_base = 50;
                break;
            case "potato":
                t_base = 44.6;
                t_min = 44.6; // NEED TO ASK ABOUT MIN AND MAX TEMPS IN DAY. SHOULD T_MIN BE SET EQUAL TO T_BASE IF IT IS LESS THAN T_BASE?
                break;
            case "wheat":
                t_base = 41.9;
                t_min = 41.9;
                break;
            case "peas":
                t_base = 41;
                t_min = 41;
                break;
            case "brussels_sprout":
            case "parsley":
            case "cabbage":
                t_base = 32;
                t_min = 32;
                break;
            default:
                errors.push({
                    parameter_error: "product",
                    message: product + " is not available for GDD calculations"
                });
                break;
        }
    }

    if (longitude < -125 || longitude > -66.5) {
        errors.push({
            parameter_error: "longitude",
            message: longitude.toString() + " is out of bounds for GDD calculations. Must be between -125.0 - -66.5"
        });
    }

    if (errors.length > 0) {
        res.status(400).send({
            errors: errors
        })
    }
    if (year != new Date().getFullYear()) {
        gdd.findOne(query, projection).then(function(data) {
            var min_temps = data["min_temps"]
            var max_temps = data["max_temps"]
            var gdds = [];
            
            var min_temp = 0
            var max_temp = 0
    
            for (var i = 0; i < min_temps.length; i++) {
                min_temp = min_temps[i] >= t_min ? min_temps[i] : t_min;
                max_temp = max_temps[i] <= t_max ? max_temps[i] : t_max;
                gdds.push(((max_temp + min_temp) / 2) - t_base)
            }
    
            res.json({
                message: "GDDs",
                date: data["last_date"],
                data: gdds,
                closest_lon: data["location"]["coordinates"][0],
                closest_lat: data["location"]["coordinates"][1]
            })
    
        }, function(err) {
            console.log(err);
        })
    } else {
        gdd_current.findOne(query, projection).then(function(data) {
            var min_temps = data["min_temps"]
            var max_temps = data["max_temps"]
            var gdds = [];
            
            var min_temp = 0
            var max_temp = 0
    
            for (var i = 0; i < min_temps.length; i++) {
                min_temp = min_temps[i] >= t_min ? min_temps[i] : t_min;
                max_temp = max_temps[i] <= t_max ? max_temps[i] : t_max;
                gdds.push(((max_temp + min_temp) / 2) - t_base)
            }
    
            res.json({
                message: "GDDs",
                date: data["last_date"],
                data: gdds,
                closest_lon: data["location"]["coordinates"][0],
                closest_lat: data["location"]["coordinates"][1]
            })
    
        }, function(err) {
            console.log(err);
        })
    }
   
};

exports.normal = function (req, res) {

    var product = req.params.product;

    if (typeof req.body.latitude === "undefined" || typeof req.body.longitude === "undefined") {
        res.status(400).send({"parameter error": "missing latitude or longitude"})
    }

    var latitude = parseFloat(req.body.latitude)
    var longitude = parseFloat(req.body.longitude)
    
    var query = {
        location: { 
            "$near": {
                "$geometry": {
                    "type": "Point", 
                    "coordinates": [longitude, latitude]
                },
            },
        },
    }

    var projection = {
        min_temps: 1,
        max_temps: 1,
        location: 1,
    }
    
    var t_base = 50
    var t_max = 86
    var t_min = 50 

    errors = []

    if (latitude < 24.083334 || latitude > 49.916668) {
        errors.push({
            parameter_error: "latitude",
            message: latitude.toString() + " is out of bounds for GDD calculations. Must be between 24.083334 - 49.916668"
        });
    }

    if (req.body.hasOwnProperty("t_base")) {
        t_base = parseFloat(req.body.t_base);
        if (t_base < t_min) {
            t_min = t_base;
        }
    } else {
        switch (product) {
            case "soybean":
            case "corn":
            case "sunflower":
            case "tomato":
            case "sugar_beat":
                t_base = 50;
                break;
            case "potato":
                t_base = 44.6;
                t_min = 44.6; // NEED TO ASK ABOUT MIN AND MAX TEMPS IN DAY. SHOULD T_MIN BE SET EQUAL TO T_BASE IF IT IS LESS THAN T_BASE?
                break;
            case "wheat":
                t_base = 41.9;
                t_min = 41.9;
                break;
            case "peas":
                t_base = 41;
                t_min = 41;
                break;
            case "brussels_sprout":
            case "parsley":
            case "cabbage":
                t_base = 32;
                t_min = 32;
                break;
            default:
                errors.push({
                    parameter_error: "product",
                    message: product + " is not available for GDD calculations"
                });
                break;
        }
    }

    if (longitude < -125 || longitude > -66.5) {
        errors.push({
            parameter_error: "longitude",
            message: longitude.toString() + " is out of bounds for GDD calculations. Must be between -125.0 - -66.5"
        });
    }

    if (errors.length > 0) {
        res.status(400).send({
            errors: errors
        })
    }

    normals.findOne(query, projection).then(function(data) {
        var min_temps = data["min_temps"]
        var max_temps = data["max_temps"]
        var gdds = [];
        
        var min_temp = 0
        var max_temp = 0

        for (var i = 0; i < min_temps.length; i++) {
            min_temp = min_temps[i] >= t_min ? min_temps[i] : t_min;
            max_temp = max_temps[i] <= t_max ? max_temps[i] : t_max;
            gdds.push(((max_temp + min_temp) / 2) - t_base)
        }

        res.json({
            message: "30-year Normal GDDs",
            data: gdds,
            closest_lon: data["location"]["coordinates"][0],
            closest_lat: data["location"]["coordinates"][1]
        })

    }, function(err) {
        console.log(err);
    })
   
};
