package models

import (
	"strconv"
	"time"

	"gitlab.cs.umd.edu/dawn/go-backend/dawn-gdd/config"
	"gitlab.cs.umd.edu/dawn/go-backend/dawn-gdd/persistence/entities"

	validation "github.com/go-ozzo/ozzo-validation"
	"github.com/gofiber/fiber/v2"
)

type GddResponse struct {
	Product          string    `json:"product"`
	ClosestLatitude  float64   `json:"closest_latitude"`
	ClosestLongitude float64   `json:"closest_longitude"`
	GddValues        []float64 `json:"gdd_values"`
	LastDate         time.Time `json:"last_date"`
}

type GddRequest struct {
	Year       int     `json:"year"`
	Product    string  `json:"product"`
	Latitude   float64 `json:"latitude"`
	Longitude  float64 `json:"longitude"`
	Accumulate bool    `json:"accumulate"`
}

func (r GddRequest) Validate() error {
	return validation.ValidateStruct(&r,
		validation.Field(&r.Year, validation.Required, validation.Min(1981), validation.Max(time.Now().Year())),
		validation.Field(&r.Product, validation.Required, validation.Length(1, 100)),
		validation.Field(&r.Latitude, validation.Required, validation.Min(-90.0), validation.Max(90.0)),
		validation.Field(&r.Longitude, validation.Required, validation.Min(-180.0), validation.Max(180.0)),
	)
}

func (r GddRequest) ValidateNoYear() error {
	return validation.ValidateStruct(&r,
		validation.Field(&r.Product, validation.Required, validation.Length(1, 100)),
		validation.Field(&r.Latitude, validation.Required, validation.Min(-90.0), validation.Max(90.0)),
		validation.Field(&r.Longitude, validation.Required, validation.Min(-180.0), validation.Max(180.0)),
	)
}

var BuildGddRequest = func(c *fiber.Ctx) GddRequest {
	year, errYear := strconv.Atoi(c.Query("year", "0"))
	product := c.Query("product")
	latitude, errLat := strconv.ParseFloat(c.Query("latitude"), 64)
	longitude, errLon := strconv.ParseFloat(c.Query("longitude"), 64)
	accumulate, errBool := strconv.ParseBool(c.Query("accumulate", "false"))

	rNew := GddRequest{
		Year:       year,
		Product:    product,
		Latitude:   latitude,
		Longitude:  longitude,
		Accumulate: accumulate,
	}

	if errYear != nil || errLat != nil || errLon != nil || errBool != nil {
		panic(config.BAD_REQUEST)
	}

	if rNew.Year != 0 {
		if e := rNew.Validate(); e != nil {
			panic(config.BAD_REQUEST.AddLogDetails(e.Error()))
		}
	} else {
		if e := rNew.ValidateNoYear(); e != nil {
			panic(config.BAD_REQUEST.AddLogDetails(e.Error()))
		}
	}
	return rNew
}

func (r GddRequest) BuildLocation() entities.Location {
	l := entities.Location{
		Type:        "Point",
		Coordinates: []float64{r.Longitude, r.Latitude},
	}
	return l
}
