package persistence

import (
	"context"
	"strings"

	"gitlab.cs.umd.edu/dawn/dawn-go-common/common"
	"gitlab.cs.umd.edu/dawn/go-backend/dawn-gdd/config"
	"gitlab.cs.umd.edu/dawn/go-backend/dawn-gdd/models"
	"gitlab.cs.umd.edu/dawn/go-backend/dawn-gdd/persistence/entities"

	"github.com/bradfitz/slice"
	"github.com/spf13/viper"
	"go.mongodb.org/mongo-driver/bson"
	"go.mongodb.org/mongo-driver/mongo"
	"go.mongodb.org/mongo-driver/mongo/options"
)

var (
	Conn       *mongo.Client
	Ctx               = context.Background()
	ConnString string = ""
)

func CreateDBSession() error {
	ConnString = viper.GetString("db.uri") + viper.GetString("db.database")
	var err error
	Conn, err = mongo.Connect(Ctx, options.Client().
		ApplyURI(ConnString))
	if err != nil {
		return err
	}
	err = Conn.Ping(Ctx, nil)
	if err != nil {
		return err
	}
	return nil
}

func buildLocationRequest(location entities.Location, year *int) bson.D {
	var filter bson.D
	if year != nil {
		filter = bson.D{
			{Key: "location", Value: bson.D{
				{Key: "$near", Value: bson.D{
					{Key: "$geometry", Value: location},
				}},
			},
			},
			{Key: "year", Value: *year},
		}
	} else {
		filter = bson.D{
			{Key: "location", Value: bson.D{
				{Key: "$near", Value: bson.D{
					{Key: "$geometry", Value: location},
				}},
			},
			},
		}
	}
	return filter
}

func CurrentGddFindFirstByYearAndLocation(ctx common.DawnCtx, location entities.Location) entities.Gdd {
	coll := Conn.Database(viper.GetString("db.database")).Collection("gdd_current")

	filter := buildLocationRequest(location, nil)

	var g entities.Gdd
	err := coll.FindOne(Ctx, filter).Decode(&g)
	if err != nil {
		panic(config.NO_DATA_FOUND)
	}

	return g
}

func GddFindFirstByYearAndLocation(year int, location entities.Location) entities.Gdd {
	coll := Conn.Database(viper.GetString("db.database")).Collection("gdd")

	filter := buildLocationRequest(location, &year)

	var g entities.Gdd
	err := coll.FindOne(Ctx, filter).Decode(&g)
	if err != nil {
		panic(config.NO_DATA_FOUND)
	}

	return g
}

func NormalsFindFirstByYearAndLocation(location entities.Location) entities.Normal {
	coll := Conn.Database(viper.GetString("db.database")).Collection("normal_gdd")

	filter := buildLocationRequest(location, nil)

	var n entities.Normal
	err := coll.FindOne(Ctx, filter).Decode(&n)
	if err != nil {
		panic(config.NO_DATA_FOUND)
	}

	return n
}

func GefsFindAllByLocation(location entities.Location) []entities.GefsGdd {
	coll := Conn.Database(viper.GetString("db.database")).Collection("gefs")

	filter := buildLocationRequest(location, nil)

	var results []entities.GefsGdd

	options := options.Find()

	count := 10
	options.SetLimit(int64(count))

	cursor, err := coll.Find(Ctx, filter, options)
	if err != nil {
		panic(config.NO_DATA_FOUND)
	}

	for cursor.Next(context.TODO()) {

		var elem entities.GefsGdd
		err := cursor.Decode(&elem)
		if err != nil {
			panic(config.NO_DATA_FOUND)
		}

		results = append(results, elem)
	}

	slice.Sort(results[:], func(i, j int) bool {
		return results[i].Date.Time().Unix() < results[j].Date.Time().Unix()
	})

	return results
}

func CfsFindAllByLocation(location entities.Location) entities.CfsGdd {
	coll := Conn.Database(viper.GetString("db.database")).Collection("cfs")

	filter := buildLocationRequest(location, nil)

	var g entities.CfsGdd
	err := coll.FindOne(Ctx, filter).Decode(&g)
	if err != nil {
		panic(config.NO_DATA_FOUND)
	}

	return g
}

func FindAnalogYear(location entities.Location) models.AnalogResponse {
	coll := Conn.Database(viper.GetString("db.database")).Collection("gdd_current")

	filter := buildLocationRequest(location, nil)

	var g entities.Gdd
	err := coll.FindOne(Ctx, filter).Decode(&g)
	if err != nil {
		panic(config.NO_DATA_FOUND)
	}

	results := models.AnalogResponse{
		ClosestLatitude:  g.Location.Coordinates[1],
		ClosestLongitude: g.Location.Coordinates[0],
		AnalogYear:       g.AnalogYear,
	}

	return results
}

func FindFreezingDates(location entities.Location) entities.FreezingDates {
	coll := Conn.Database(viper.GetString("db.database")).Collection("freezing_dates")

	filter := buildLocationRequest(location, nil)

	var g entities.FreezingDates
	err := coll.FindOne(Ctx, filter).Decode(&g)
	if err != nil {
		panic(config.NO_DATA_FOUND)
	}

	return g
}

func FindSeed(seedName string) entities.Seed {
	coll := Conn.Database(viper.GetString("db.database")).Collection("seeds")

	filter := bson.D{
		{Key: "seed", Value: seedName},
	}

	var g entities.Seed
	err := coll.FindOne(Ctx, filter).Decode(&g)
	if err != nil {
		panic(config.NO_DATA_FOUND)
	}

	return g
}

func FindSeeds(product string) []entities.Seed {

	coll := Conn.Database(viper.GetString("db.database")).Collection("seeds")

	filter := bson.D{
		{Key: "type", Value: strings.ToLower(product)},
	}

	var results []entities.Seed

	options := options.Find()

	cursor, err := coll.Find(Ctx, filter, options)
	if err != nil {
		panic(config.NO_DATA_FOUND)
	}

	for cursor.Next(context.TODO()) {

		var elem entities.Seed
		err := cursor.Decode(&elem)
		if err != nil {
			panic(config.NO_DATA_FOUND)
		}

		results = append(results, elem)
	}

	return results
}
