package services

import (
	"math"

	"gitlab.cs.umd.edu/dawn/go-backend/dawn-gdd/models"
	"gitlab.cs.umd.edu/dawn/go-backend/dawn-gdd/models/enums"
	"gitlab.cs.umd.edu/dawn/go-backend/dawn-gdd/persistence"
	"gitlab.cs.umd.edu/dawn/go-backend/dawn-gdd/utils"
)

func GetGefsGddValues(request models.GddRequest) models.GefsGddResponse {
	product := enums.GetProductFromString(request.Product)
	location := request.BuildLocation()
	g := persistence.GefsFindAllByLocation(location)
	var returnGdds models.GefsGddResponse

	var gdds []float64
	var lowerBound []float64
	var upperBound []float64

	for i := 0; i < 10; i++ {
		temp := g[i]
		variance := (temp.VarMin + temp.VarMax)
		variance *= math.Pow(0.5, 2)
		std := math.Pow(variance, 0.5)

		value := utils.CalculateSingleGdd(temp.MinTemp, temp.MaxTemp, product)

		lowerBoundValue := (value - (std / (math.Pow(3, 0.5)) * 1.960))
		upperBoundValue := (value + (std / (math.Pow(3, 0.5)) * 1.960))

		lowerBoundValue = utils.ClipMinFloat(lowerBoundValue, 0.0)
		upperBoundValue = utils.ClipMinFloat(upperBoundValue, 0.0)

		if request.Accumulate && i > 0 {
			value += gdds[len(gdds)-1]
			lowerBoundValue += lowerBound[len(gdds)-1]
			upperBoundValue += upperBound[len(gdds)-1]
		}
		gdds = append(gdds, value)
		lowerBound = append(lowerBound, lowerBoundValue)
		upperBound = append(upperBound, upperBoundValue)
	}

	returnGdds = models.GefsGddResponse{
		Product:          product.Name,
		ClosestLatitude:  location.Coordinates[1],
		ClosestLongitude: location.Coordinates[0],
		GddValues:        gdds,
		UpperBound:       upperBound,
		LowerBound:       lowerBound,
		FirstDate:        g[0].Date.Time().UTC(),
		LastDate:         g[9].Date.Time().UTC(),
	}
	return returnGdds
}

func GetCfsGddValues(request models.GddRequest) models.CfsGddResponse {
	product := enums.GetProductFromString(request.Product)
	location := request.BuildLocation()
	g := persistence.CfsFindAllByLocation(location)
	var returnGdds models.CfsGddResponse

	var gdds []float64
	var lowerBound []float64
	var upperBound []float64

	for i := 0; i < len(g.MaxTemps); i++ {

		variance := (g.VarMin[i] + g.VarMax[i])
		variance *= math.Pow(0.5, 2)
		std := math.Pow(variance, 0.5)

		value := utils.CalculateSingleGdd(g.MinTemps[i], g.MaxTemps[i], product)

		lowerBoundValue := (value - (std / (math.Pow(4, 0.5)) * 1.960))
		upperBoundValue := (value + (std / (math.Pow(4, 0.5)) * 1.960))

		lowerBoundValue = utils.ClipMinFloat(lowerBoundValue, 0.0)
		upperBoundValue = utils.ClipMinFloat(upperBoundValue, 0.0)

		if request.Accumulate && i > 0 {
			value += gdds[len(gdds)-1]
			lowerBoundValue += lowerBound[len(gdds)-1]
			upperBoundValue += upperBound[len(gdds)-1]
		}
		gdds = append(gdds, value)
		lowerBound = append(lowerBound, lowerBoundValue)
		upperBound = append(upperBound, upperBoundValue)
	}

	returnGdds = models.CfsGddResponse{
		Product:          product.Name,
		ClosestLatitude:  location.Coordinates[1],
		ClosestLongitude: location.Coordinates[0],
		GddValues:        gdds,
		LowerBound:       lowerBound,
		UpperBound:       upperBound,
		FirstDate:        g.Date.Time().UTC(),
	}
	return returnGdds
}
