package services

import (
	"math"

	"gitlab.cs.umd.edu/dawn/go-backend/dawn-gdd/models"
	"gitlab.cs.umd.edu/dawn/go-backend/dawn-gdd/models/enums"
	"gitlab.cs.umd.edu/dawn/go-backend/dawn-gdd/persistence"
	"gitlab.cs.umd.edu/dawn/go-backend/dawn-gdd/utils"
	"gonum.org/v1/gonum/stat"
)

func GetGefsGddValues(request models.GddRequest) models.GefsGddResponse {
	product := enums.GetProductFromString(request.Product)
	location := request.BuildLocation()
	g := persistence.GefsFindAllByLocation(location)
	var returnGdds models.GefsGddResponse

	gdds := []float64{}
	lowerBound := []float64{}
	upperBound := []float64{}

	// get the int of the planting date. If the date is less than the first date, we do nothing
	// otherwise, we adjust
	// need to do before because of accumulations
	sliceDateInt := utils.DaysSince(g[0].Date.Time(), request.PlantingDate)
	fd := request.PlantingDate
	if sliceDateInt < 0 {
		sliceDateInt = 0
		fd = g[0].Date.Time()
	}

	for i := sliceDateInt; i < 10; i++ {
		temp := g[i]
		variance := (temp.VarMin + temp.VarMax)
		variance *= math.Pow(0.5, 2)
		std := math.Pow(variance, 0.5)

		value := utils.CalculateSingleGdd(temp.MinTemp, temp.MaxTemp, product)

		lowerBoundValue := (value - (std / (math.Pow(3, 0.5)) * 1.960))
		upperBoundValue := (value + (std / (math.Pow(3, 0.5)) * 1.960))

		lowerBoundValue = utils.ClipMinFloat(lowerBoundValue, 0.0)
		upperBoundValue = utils.ClipMinFloat(upperBoundValue, 0.0)

		if request.Accumulate && i > sliceDateInt {
			value += gdds[len(gdds)-1]
			lowerBoundValue += lowerBound[len(gdds)-1]
			upperBoundValue += upperBound[len(gdds)-1]
		}
		gdds = append(gdds, value)
		lowerBound = append(lowerBound, lowerBoundValue)
		upperBound = append(upperBound, upperBoundValue)
	}

	returnGdds = models.GefsGddResponse{
		Product:          product.Name,
		ClosestLatitude:  location.Coordinates[1],
		ClosestLongitude: location.Coordinates[0],
		GddValues:        gdds,
		UpperBound:       upperBound,
		LowerBound:       lowerBound,
		FirstDate:        fd.UTC(),
		LastDate:         g[9].Date.Time().UTC(),
	}
	return returnGdds
}

func GetCfsGddValues(request models.GddRequest) models.CfsGddResponse {
	location := request.BuildLocation()
	gs := persistence.CfsFindByLocation(location)
	product := enums.GetProductFromString(request.Product)
	var returnGdds models.CfsGddResponse

	gdds := []float64{}
	lowerBound := []float64{}
	upperBound := []float64{}

	// get the int of the planting date. If the date is less than the first date, we do nothing
	// otherwise, we adjust
	// need to do before because of accumulations
	sliceDateInt := utils.DaysSince(gs[0].Date.Time(), request.PlantingDate)
	fd := request.PlantingDate
	if sliceDateInt < 0 {
		sliceDateInt = 0
		fd = gs[0].Date.Time()
	}

	rows := [][]float64{}
	for i := sliceDateInt; i < len(gs[0].MinTemps); i++ {
		row := []float64{}

		for j := 0; j < len(gs); j++ {
			row = append(row, utils.CalculateSingleGdd(gs[j].MinTemps[i], gs[j].MaxTemps[i], product))
			if request.Accumulate && i > sliceDateInt {
				row[j] += rows[len(rows)-1][j]
			}
		}

		rows = append(rows, row)
		mean, std := stat.MeanStdDev(row, nil)
		gdds = append(gdds, mean)
		upperBound = append(upperBound, utils.ClipMinFloat(mean+std*1.645, 0))
		lowerBound = append(lowerBound, utils.ClipMinFloat(mean-std*1.645, 0))
	}

	returnGdds = models.CfsGddResponse{
		Product:          product.Name,
		ClosestLatitude:  location.Coordinates[1],
		ClosestLongitude: location.Coordinates[0],
		GddValues:        gdds,
		LowerBound:       lowerBound,
		UpperBound:       upperBound,
		FirstDate:        fd.UTC(),
	}
	return returnGdds
}
