package controllers

import (
	"testing"

	"github.com/gofiber/fiber/v2"
	"github.com/stretchr/testify/assert"
	DawnTest "github.com/tgs266/dawn-go-common/testing"
	"gitlab.cs.umd.edu/dawn/go-backend/dawn-gdd/models"
	"gitlab.cs.umd.edu/dawn/go-backend/dawn-gdd/persistence"
	"gitlab.cs.umd.edu/dawn/go-backend/dawn-gdd/persistence/entities"
	"gitlab.cs.umd.edu/dawn/go-backend/dawn-gdd/services"
)

func mockFindAnalogYear(location entities.Location) models.AnalogResponse {
	return models.AnalogResponse{
		ClosestLatitude:  4.0,
		ClosestLongitude: 4.0,
		AnalogYear:       2001,
	}
}

func mockGetConfidenceInterval(request models.ConfidenceIntervalRequest) models.ConfidenceIntervalResposne {
	return models.ConfidenceIntervalResposne{
		ClosestLatitude:  request.Latitude,
		ClosestLongitude: request.Longitude,
		LowerBound:       []float64{1.0, 1.0, 2.0, 2.0},
		UpperBound:       []float64{5.0, 5.0, 10.0, 10.0},
	}
}

func TestGetAnalogYear(t *testing.T) {
	mock := DawnTest.CreateMock(persistence.FindAnalogYear, mockFindAnalogYear)
	defer mock.Unpatch()
	app := fiber.New()
	app.Get("/test", GetAnalogYear)
	var data models.AnalogResponse

	params := make(map[string]string)
	params["latitude"] = "12.0"
	params["longitude"] = "12.0"

	statusCode := DawnTest.TestGetRequestParams(app, "/test", params, "token", &data)

	DawnTest.StatusCodeEqual(t, 200, statusCode)
	assert.Equal(t, 2001, data.AnalogYear, "should match")
}

func TestGetConfidenceInterval(t *testing.T) {
	mock := DawnTest.CreateMock(services.GetConfidenceInterval, mockGetConfidenceInterval)
	defer mock.Unpatch()
	app := fiber.New()
	app.Get("/test", GetConfidenceInterval)
	var data models.ConfidenceIntervalResposne

	params := make(map[string]string)
	params["product"] = "corn"
	params["latitude"] = "12.0"
	params["longitude"] = "12.0"
	params["interval"] = "95"

	statusCode := DawnTest.TestGetRequestParams(app, "/test", params, "token", &data)

	DawnTest.StatusCodeEqual(t, 200, statusCode)
	assert.Equal(t, 1.0, data.LowerBound[1], "should match")
	assert.Equal(t, 10.0, data.UpperBound[2], "should match")
}
