package persistence

import (
	"context"
	"strings"

	"github.com/tgs266/dawn-go-common/common"
	"gitlab.cs.umd.edu/dawn/go-backend/dawn-gdd/config"
	"gitlab.cs.umd.edu/dawn/go-backend/dawn-gdd/models"
	"gitlab.cs.umd.edu/dawn/go-backend/dawn-gdd/persistence/entities"

	"github.com/bradfitz/slice"
	"go.mongodb.org/mongo-driver/bson"
	"go.mongodb.org/mongo-driver/bson/primitive"
	"go.mongodb.org/mongo-driver/mongo/options"
)

var Session *common.DBSession

func buildLocationRequest(location entities.Location, year *int) bson.M {
	var filter bson.M
	if year != nil {
		filter = bson.M{
			"location": bson.M{
				"$nearSphere": bson.M{
					"$geometry":    location,
					"$maxDistance": 50000,
				},
			},
			"year": *year,
		}
	} else {
		filter = bson.M{
			"location": bson.M{
				"$nearSphere": bson.M{
					"$geometry":    location,
					"$maxDistance": 50000,
				},
			},
		}
	}
	return filter
}

func buildLocationRequestLarger(location entities.Location, year *int) bson.M {
	var filter bson.M
	if year != nil {
		filter = bson.M{
			"location": bson.M{
				"$nearSphere": bson.M{
					"$geometry":    location,
					"$maxDistance": 100000,
				},
			},
			"year": *year,
		}
	} else {
		filter = bson.M{
			"location": bson.M{
				"$nearSphere": bson.M{
					"$geometry":    location,
					"$maxDistance": 100000,
				},
			},
		}
	}
	return filter
}

func CurrentGddFindFirstByYearAndLocation(ctx common.DawnCtx, location entities.Location) entities.Gdd {
	coll := Session.Collection("gdd_current")

	filter := buildLocationRequest(location, nil)

	var g entities.Gdd
	err := coll.FindOne(Session.Ctx, filter).Decode(&g)
	if err != nil {
		panic(config.NO_DATA_FOUND.PutDetail("reason", "gdd_current"))
	}

	return g
}

func GddFindFirstByYearAndLocation(year int, location entities.Location) entities.Gdd {
	coll := Session.Collection("gdd")

	filter := buildLocationRequest(location, &year)
	var g entities.Gdd
	err := coll.FindOne(Session.Ctx, filter).Decode(&g)
	if err != nil {
		panic(config.NO_DATA_FOUND.PutDetail("reason", "gdd"))
	}

	return g
}

func NormalsFindFirstByYearAndLocation(location entities.Location) entities.Normal {
	coll := Session.Collection("normal_gdd")

	filter := buildLocationRequest(location, nil)

	var n entities.Normal
	err := coll.FindOne(Session.Ctx, filter).Decode(&n)
	if err != nil {
		panic(config.NO_DATA_FOUND.PutDetail("reason", "normal_gdd"))
	}

	return n
}

func GetLastNormalsYearly(location entities.Location) []entities.Gdd {
	coll := Session.Collection("gdd")

	filter := buildLocationRequest(location, nil)
	filter["year"] = bson.M{"$gte": 1991, "$lte": 2020}

	var n []entities.Gdd
	cursor, err := coll.Find(Session.Ctx, filter, options.Find().SetLimit(30))
	if err != nil {
		panic(config.NO_DATA_FOUND.PutDetail("reason", "gdd"))
	}

	if err = cursor.All(Session.Ctx, &n); err != nil {
		panic(config.INTERNAL_SERVER_STANDARD_ERROR.AddLogDetails(err.Error()))
	}

	if len(n) == 0 {
		panic(config.NO_DATA_FOUND.PutDetail("reason", "gdd"))
	}

	return n
}

func GefsFindAllByLocation(location entities.Location) []entities.GefsGdd {
	coll := Session.Collection("gefs")

	filter := buildLocationRequestLarger(location, nil)

	var results []entities.GefsGdd

	options := options.Find()

	count := 10
	options.SetLimit(int64(count))

	cursor, err := coll.Find(Session.Ctx, filter, options)
	if err != nil {
		panic(config.NO_DATA_FOUND.PutDetail("reason", "gefs"))
	}

	for cursor.Next(context.TODO()) {

		var elem entities.GefsGdd
		err := cursor.Decode(&elem)
		if err != nil {
			panic(config.NO_DATA_FOUND.PutDetail("reason", "gefs"))
		}

		results = append(results, elem)
	}

	slice.Sort(results[:], func(i, j int) bool {
		return results[i].Date.Time().Unix() < results[j].Date.Time().Unix()
	})

	return results
}

func CfsFindAllByLocation(location entities.Location) entities.CfsGdd {
	coll := Session.Collection("cfs")

	filter := buildLocationRequestLarger(location, nil)
	filter["member"] = -1
	var g entities.CfsGdd
	err := coll.FindOne(Session.Ctx, filter).Decode(&g)

	if err != nil {
		panic(config.NO_DATA_FOUND.PutDetail("reason", "cfs").AddLogDetails(err.Error()))
	}

	return g
}

func CfsFindByLocation(location entities.Location) []entities.CfsGdd {
	cfs := CfsFindAllByLocation(location)

	coll := Session.Collection("cfs")

	filter := buildLocationRequestLarger(location, nil)
	// cursor, err := coll.Find(Session.Ctx, filter, options.Find().SetLimit(int64(2)))
	cursor, err := coll.Find(Session.Ctx, filter, options.Find().SetLimit(int64(cfs.Count)))

	if err != nil {
		panic(config.NO_DATA_FOUND.PutDetail("reason", "cfs").AddLogDetails(err.Error()))
	}

	var gs []entities.CfsGdd
	if err = cursor.All(Session.Ctx, &gs); err != nil {
		panic(config.INTERNAL_SERVER_STANDARD_ERROR.AddLogDetails(err.Error()))
	}

	return gs
}

func CfsFindByLocationMultiple(location entities.Location, mult int) []entities.CfsGdd {
	cfs := CfsFindAllByLocation(location)

	coll := Session.Collection("cfs")

	filter := buildLocationRequestLarger(location, nil)
	// cursor, err := coll.Find(Session.Ctx, filter, options.Find().SetLimit(int64(2)))
	cursor, err := coll.Find(Session.Ctx, filter, options.Find().SetLimit(int64(cfs.Count)*int64(mult)))

	if err != nil {
		panic(config.NO_DATA_FOUND.PutDetail("reason", "cfs").AddLogDetails(err.Error()))
	}

	var gs []entities.CfsGdd
	if err = cursor.All(Session.Ctx, &gs); err != nil {
		panic(config.INTERNAL_SERVER_STANDARD_ERROR.AddLogDetails(err.Error()))
	}

	return gs
}

func FindAnalogYear(location entities.Location) models.AnalogResponse {
	coll := Session.Collection("gdd_current")

	filter := buildLocationRequestLarger(location, nil)

	var g entities.Gdd
	err := coll.FindOne(Session.Ctx, filter).Decode(&g)
	if err != nil {
		panic(config.NO_DATA_FOUND)
	}

	results := models.AnalogResponse{
		ClosestLatitude:  g.Location.Coordinates[1],
		ClosestLongitude: g.Location.Coordinates[0],
		AnalogYear:       g.AnalogYear,
	}

	return results
}

func FindFreezingDates(location entities.Location) entities.FreezingDates {
	coll := Session.Collection("freezing_dates")

	filter := buildLocationRequest(location, nil)

	var g entities.FreezingDates
	err := coll.FindOne(Session.Ctx, filter).Decode(&g)
	if err != nil {
		panic(config.NO_DATA_FOUND)
	}

	return g
}

func FindSeed(seedName string) entities.Seed {
	coll := Session.Collection("seeds")

	filter := bson.D{
		{Key: "seed", Value: seedName},
	}

	var g entities.Seed
	err := coll.FindOne(Session.Ctx, filter).Decode(&g)
	if err != nil {
		panic(config.NO_DATA_FOUND)
	}

	return g
}

func FindSeeds(product string) []entities.Seed {

	coll := Session.Collection("seeds")

	filter := bson.D{
		{Key: "type", Value: strings.ToLower(product)},
	}

	var results []entities.Seed

	options := options.Find()

	cursor, err := coll.Find(Session.Ctx, filter, options)
	if err != nil {
		panic(config.NO_DATA_FOUND)
	}

	for cursor.Next(context.TODO()) {

		var elem entities.Seed
		err := cursor.Decode(&elem)
		if err != nil {
			panic(config.NO_DATA_FOUND)
		}

		results = append(results, elem)
	}

	return results
}

func FindCultivarByName(name string) entities.Cultivar {
	coll := Session.Collection("dssat")

	filter := bson.M{
		"var_name": bson.M{
			"$regex": primitive.Regex{Pattern: name, Options: "i"},
		},
	}

	var results entities.Cultivar

	err := coll.FindOne(Session.Ctx, filter).Decode(&results)
	if err != nil {
		panic(config.NO_DATA_FOUND)
	}

	return results
}

func FindCultivarsByName() []entities.Cultivar {
	coll := Session.Collection("dssat")

	var results []entities.Cultivar

	options := options.Find()

	cursor, err := coll.Find(Session.Ctx, bson.D{}, options)
	if err != nil {
		panic(config.NO_DATA_FOUND)
	}

	for cursor.Next(context.TODO()) {

		var elem entities.Cultivar
		err := cursor.Decode(&elem)
		if err != nil {
			panic(config.NO_DATA_FOUND)
		}

		results = append(results, elem)
	}

	return results
}
