package cmd

import (
	"fmt"
	"os"
	"os/signal"
	"runtime"
	"strconv"
	"strings"
	"syscall"

	"github.com/tgs266/dawn-go-common/common"
	"gitlab.cs.umd.edu/dawn/go-backend/dawn-gdd/controllers"
	"gitlab.cs.umd.edu/dawn/go-backend/dawn-gdd/docs"
	"gitlab.cs.umd.edu/dawn/go-backend/dawn-gdd/persistence"

	"github.com/ansrivas/fiberprometheus/v2"
	swagger "github.com/arsmn/fiber-swagger/v2"
	"github.com/gofiber/fiber/v2"
	"github.com/gofiber/fiber/v2/middleware/cors"
	"github.com/gofiber/fiber/v2/middleware/recover"
	"github.com/gofiber/fiber/v2/middleware/requestid"
	"github.com/gofiber/fiber/v2/utils"
	"github.com/spf13/viper"

	_ "gitlab.cs.umd.edu/dawn/go-backend/dawn-gdd/docs"
)

func registerRoutes(app *fiber.App) {
	api := app.Group(viper.GetString("server.context-path"))
	controllers.GddRoutes(api)
}

func registerSwagger(app *fiber.App) {
	if viper.GetBool("app.swagger") {
		app.Get(viper.GetString("server.context-path")+"/swagger/*", swagger.Handler)
		app.Get(viper.GetString("server.context-path")+"/swagger/*", swagger.New())
	}
}

func registerLogging(app *fiber.App) {
	app.Use(requestid.New(requestid.Config{
		Next:   nil,
		Header: fiber.HeaderXRequestID,
		Generator: func() string {
			return utils.UUIDv4()
		},
		ContextKey: "requestId",
	}))

	app.Use(common.FiberLogger())
}

func registerPrometheus(app *fiber.App) {
	prom := fiberprometheus.New(viper.GetString("app.name"))
	prom.RegisterAt(app, viper.GetString("server.context-path")+"/metrics")
	app.Use(prom.Middleware)

	common.RegisterDawnPrometheus()
}

func registerCors(app *fiber.App) {
	app.Use(cors.New())
}

func createFiberConfig() fiber.Config {
	return fiber.Config{
		ErrorHandler: common.DawnErrorHandler,
	}
}

func CreateFiberApp() *fiber.App {

	app := fiber.New(createFiberConfig())
	app.Use(recover.New(recover.Config{
		EnableStackTrace: true,
		StackTraceHandler: func(c *fiber.Ctx, e interface{}) {
			buf := make([]byte, 1024)
			buf = buf[:runtime.Stack(buf, false)]
			c.Locals("stack_trace", string(buf))
		},
	}))
	registerCors(app)
	registerSwagger(app)
	registerLogging(app)
	registerPrometheus(app)
	app.Use(common.FiberLoadBalanceInsert())
	common.RegisterHealth(app)
	registerRoutes(app)

	if viper.GetBool("app.messaging") {
		common.RegisterHeartbeatPublisher()
	}

	return app
}

// @title Dawn GDD Service
// @description All operations for GDD/Freezing Date data
// @contact.name API Support
// @contact.email tgsiegel@umd.edu
func start(path string) {
	common.ClearLogFolder()
	common.GetConfig(path)

	apiVersion := strconv.Itoa(viper.GetInt("app.api-version"))

	contextPath := viper.GetString("server.context-path")
	contextPath = strings.ReplaceAll(contextPath, "/api/", "/api/v"+apiVersion+"/")
	viper.Set("server.context-path", contextPath)

	persistence.Session, _ = common.CreateDBSession(viper.GetString("db.database"))

	docs.SwaggerInfo.Host = viper.GetString("app.swagger-host-url")
	docs.SwaggerInfo.BasePath = viper.GetString("server.context-path")
	docs.SwaggerInfo.Version = "V" + apiVersion

	app := CreateFiberApp()

	c := make(chan os.Signal, 1)
	signal.Notify(c, os.Interrupt)
	signal.Notify(c, syscall.SIGTERM)
	go func() {
		_ = <-c
		fmt.Println("Gracefully shutting down...")
		_ = app.Shutdown()
	}()

	err2 := app.Listen(":" + viper.GetString("server.port"))

	if err2 != nil {
		panic(err2)
	}
}
