package controllers

import (
	"testing"
	"time"

	"github.com/gofiber/fiber/v2"
	"github.com/stretchr/testify/assert"
	"github.com/tgs266/dawn-go-common/common"
	DawnTest "github.com/tgs266/dawn-go-common/testing"
	"gitlab.cs.umd.edu/dawn/go-backend/dawn-gdd/models"
	"gitlab.cs.umd.edu/dawn/go-backend/dawn-gdd/services"
)

func mockGetGddValues(ctx common.DawnCtx, request models.GddRequest) models.GddResponse {
	return models.GddResponse{
		Product:          request.Product,
		ClosestLatitude:  request.Latitude,
		ClosestLongitude: request.Longitude,
		GddValues:        []float64{1.0, 2.0, 3.0, 4.0},
		LastDate:         time.Date(2021, time.January, 4, 0, 0, 0, 0, time.Now().Location()),
	}
}

func mockGetNormalValues(request models.GddRequest) models.GddResponse {
	return models.GddResponse{
		Product:          request.Product,
		ClosestLatitude:  request.Latitude,
		ClosestLongitude: request.Longitude,
		GddValues:        []float64{1.0, 2.0, 3.0, 4.0},
		LastDate:         time.Date(2021, time.January, 4, 0, 0, 0, 0, time.Now().Location()),
	}
}

func TestGetDailyGdd(t *testing.T) {
	mock := DawnTest.CreateMock(services.GetGddValues, mockGetGddValues)
	defer mock.Unpatch()
	app := fiber.New()
	app.Get("/test", GetDailyGdd)
	var data models.GddResponse

	params := make(map[string]string)
	params["year"] = "2021"
	params["product"] = "corn"
	params["latitude"] = "12.0"
	params["longitude"] = "12.0"
	params["accumulate"] = "false"

	statusCode := DawnTest.TestGetRequestParams(app, "/test", params, "token", &data)

	DawnTest.StatusCodeEqual(t, 200, statusCode)
	assert.Equal(t, "corn", data.Product, "should match")
	assert.Equal(t, 4.0, data.GddValues[3], "should match")
}

func TestGetNormalGdd(t *testing.T) {
	mock := DawnTest.CreateMock(services.GetNormalValues, mockGetNormalValues)
	defer mock.Unpatch()
	app := fiber.New()
	app.Get("/test", GetNormalGdd)
	var data models.GddResponse

	params := make(map[string]string)
	params["product"] = "corn"
	params["latitude"] = "12.0"
	params["longitude"] = "12.0"
	params["accumulate"] = "false"

	statusCode := DawnTest.TestGetRequestParams(app, "/test", params, "token", &data)

	DawnTest.StatusCodeEqual(t, 200, statusCode)
	assert.Equal(t, "corn", data.Product, "should match")
	assert.Equal(t, 4.0, data.GddValues[3], "should match")
}
