package controllers

import (
	"testing"
	"time"

	"github.com/gofiber/fiber/v2"
	"github.com/stretchr/testify/assert"
	DawnTest "github.com/tgs266/dawn-go-common/testing"
	"gitlab.cs.umd.edu/dawn/go-backend/dawn-gdd/models"
	"gitlab.cs.umd.edu/dawn/go-backend/dawn-gdd/services"
)

func mockGetGefsGddValues(request models.GddRequest) models.GefsGddResponse {
	return models.GefsGddResponse{
		Product:          request.Product,
		ClosestLatitude:  request.Latitude,
		ClosestLongitude: request.Longitude,
		GddValues:        []float64{1.0, 2.0, 3.0, 4.0},
		UpperBound:       []float64{1.0, 2.0, 3.0, 4.0},
		LowerBound:       []float64{1.0, 2.0, 3.0, 4.0},
		FirstDate:        time.Date(2021, time.January, 1, 0, 0, 0, 0, time.Now().Location()),
		LastDate:         time.Date(2021, time.January, 4, 0, 0, 0, 0, time.Now().Location()),
	}
}

func mockGetCfsGddValues(request models.GddRequest) models.CfsGddResponse {
	return models.CfsGddResponse{
		Product:          request.Product,
		ClosestLatitude:  request.Latitude,
		ClosestLongitude: request.Longitude,
		GddValues:        []float64{1.0, 2.0, 3.0, 4.0},
		UpperBound:       []float64{1.0, 2.0, 3.0, 4.0},
		LowerBound:       []float64{1.0, 2.0, 3.0, 4.0},
		FirstDate:        time.Date(2021, time.January, 1, 0, 0, 0, 0, time.Now().Location()),
	}
}

func TestGetGefsGDD(t *testing.T) {
	mock := DawnTest.CreateMock(services.GetGefsGddValues, mockGetGefsGddValues)
	defer mock.Unpatch()
	app := fiber.New()
	app.Get("/test", GetGefsGDD)
	var data models.GddResponse

	params := make(map[string]string)
	params["year"] = "2021"
	params["product"] = "corn"
	params["latitude"] = "12.0"
	params["longitude"] = "12.0"
	params["accumulate"] = "false"

	statusCode := DawnTest.TestGetRequestParams(app, "/test", params, "token", &data)

	DawnTest.StatusCodeEqual(t, 200, statusCode)
	assert.Equal(t, "corn", data.Product, "should match")
	assert.Equal(t, 4.0, data.GddValues[3], "should match")
}

func TestGetCfsGDD(t *testing.T) {
	mock := DawnTest.CreateMock(services.GetCfsGddValues, mockGetCfsGddValues)
	defer mock.Unpatch()
	app := fiber.New()
	app.Get("/test", GetCfsGDD)
	var data models.GddResponse

	params := make(map[string]string)
	params["year"] = "2021"
	params["product"] = "corn"
	params["latitude"] = "12.0"
	params["longitude"] = "12.0"
	params["accumulate"] = "false"

	statusCode := DawnTest.TestGetRequestParams(app, "/test", params, "token", &data)

	DawnTest.StatusCodeEqual(t, 200, statusCode)
	assert.Equal(t, "corn", data.Product, "should match")
	assert.Equal(t, 4.0, data.GddValues[3], "should match")
}
