package models

import (
	"strconv"
	"strings"
	"time"

	validation "github.com/go-ozzo/ozzo-validation"

	"github.com/tgs266/dawn-go-common/common"
	"gitlab.cs.umd.edu/dawn/go-backend/dawn-gdd/config"
)

type StageRequest struct {
	PlantDate   time.Time `json:"plant_date"`
	Mode        string    `json:"mode"`
	Value       int       `json:"value"`
	Latitude    float64   `json:"latitude"`
	Longitude   float64   `json:"longitude"`
	AnchorStage string    `json:"anchor_stage"`
	AnchorDate  time.Time `json:"anchor_date"`
	Comparison  int       `json:"comparison"`
}

func BuildStageRequest(ctx common.DawnCtx) StageRequest {
	plantDate := ctx.FiberCtx.Query("plant_date")
	mode := ctx.FiberCtx.Query("mode")
	value := ctx.FiberCtx.Query("value")
	latitude := ctx.FiberCtx.Query("latitude")
	longitude := ctx.FiberCtx.Query("longitude")
	anchorStage := ctx.FiberCtx.Query("anchor_stage")
	anchorDateString := ctx.FiberCtx.Query("anchor_date")
	comparison := ctx.FiberCtx.Query("comparison", "-1")

	pd, err := time.Parse(time.RFC3339, plantDate)
	if err != nil {
		panic(config.BAD_REQUEST.PutDetail("reason", err.Error()))
	}

	anchorDate := time.Time{}
	if anchorStage != "" {
		anchorDate, err = time.Parse(time.RFC3339, anchorDateString)
		if err != nil {
			panic(config.BAD_REQUEST.PutDetail("reason", err.Error()))
		}
	}

	modeLower := strings.ToLower(mode)
	if modeLower != "rm" && modeLower != "gdds_to_maturity" {
		panic(config.BAD_REQUEST.PutDetail("reason", "\""+mode+"\" is not valid"))
	}

	comparisonInt, err := strconv.Atoi(comparison)
	if err != nil {
		panic(config.BAD_REQUEST.PutDetail("reason", err.Error()))
	}

	valueInt, err := strconv.Atoi(value)
	if err != nil {
		panic(config.BAD_REQUEST.PutDetail("reason", err.Error()))
	}

	latitudeFloat, err := strconv.ParseFloat(latitude, 64)
	if err != nil {
		panic(config.BAD_REQUEST.PutDetail("reason", err.Error()))
	}

	longitudeFloat, err := strconv.ParseFloat(longitude, 64)
	if err != nil {
		panic(config.BAD_REQUEST.PutDetail("reason", err.Error()))
	}

	stageRequest := StageRequest{
		PlantDate:   pd,
		Mode:        modeLower,
		Value:       valueInt,
		Latitude:    latitudeFloat,
		Longitude:   longitudeFloat,
		AnchorDate:  anchorDate,
		AnchorStage: anchorStage,
		Comparison:  comparisonInt,
	}

	e := stageRequest.Validate()
	if e != nil {
		panic(config.BAD_REQUEST.PutDetail("reason", e.Error()))
	}

	return stageRequest
}

func (r StageRequest) Validate() error {
	return validation.ValidateStruct(&r,
		validation.Field(&r.Latitude, validation.Required, validation.Min(-90.0), validation.Max(90.0)),
		validation.Field(&r.Longitude, validation.Required, validation.Min(-180.0), validation.Max(180.0)),
	)
}

type StageData struct {
	AllGdds       [][]float64
	NormalMean    []float64
	Normal5th     []float64
	ComparisonAll [][]float64
	Normal95th    []float64
}

type Bin struct {
	Date  time.Time `json:"date"`
	Value float64   `json:"value"`
}

type Bins struct {
	Bins           []Bin     `json:"bins"`
	ComparisonMean time.Time `json:"comparison_mean"`
	Count          int       `json:"count"`
}

type StageMatches struct {
	Harvest float64
	R1Silk  float64
}

type StageStateInner struct {
	Dists []float64
	Hists []int

	NormalMeanDist float64
	Normal5thDist  float64
	Normal95thDist float64

	NormalMeanIdx int
	Normal5thIdx  int
	Normal95thIdx int
}

func (s StageStateInner) AddToAllFound(val int) StageStateInner {
	for i, _ := range s.Hists {
		s.Hists[i] += val
	}

	return s
}

func (s StageStateInner) ExtractDates(plantDate time.Time, start int) []time.Time {
	ret := []time.Time{}
	for i, _ := range s.Hists {
		ret = append(ret, plantDate.AddDate(0, 0, s.Hists[i]-start))
	}
	return ret
}

func BuildStageMatches(mode string, value int, start int, fyData StageData, req StageRequest) map[string]float64 {
	var harvestVal float64
	if mode == "rm" {
		harvestVal = float64(((float64(value) - 95.0) * 22.0) + 2375.0)
	} else {
		harvestVal = float64(value)
	}

	// if !req.AnchorDate.IsZero() {
	// 	mod := 1.0
	// 	switch req.AnchorStage {
	// 	case "emergence":
	// 		mod = 1.0 / 0.07
	// 	case "3LeafCollar":
	// 		mod = 1.0 / 0.13
	// 	case "6LeafCollar":
	// 		mod = 1.0 / 0.2
	// 	case "silk":
	// 		mod = 1.0 / 0.545
	// 	case "milk":
	// 		mod = 1.0 / 0.725
	// 	}
	// 	accMean := 0.0
	// 	for i := start; i < len(fyData.MaxGdds); i++ {
	// 		accMean += fyData.MeanGdds[i]
	// 		meanDate := req.PlantDate.AddDate(0, 0, i-start)
	// 		if req.AnchorDate == meanDate {
	// 			harvestVal = accMean * mod
	// 			break
	// 		}
	// 	}
	// }

	return map[string]float64{
		"emergence":   harvestVal * 0.07,
		"3LeafCollar": harvestVal * 0.13,
		"6LeafCollar": harvestVal * 0.2,
		"silk":        harvestVal * 0.545,
		"milk":        harvestVal * 0.725,
		"harvest":     harvestVal,
	}
}

// type StageResponse struct {

// }

type FreezingForecastRequest struct {
	Latitude     float64 `json:"latitude"`
	Longitude    float64 `json:"longitude"`
	FreezingTemp float64 `json:"freezing_temp"`
}

func BuildFreezingForecastRequest(ctx common.DawnCtx) FreezingForecastRequest {
	freezingTemp := ctx.FiberCtx.Query("freezing_temp", "0.0")
	latitude := ctx.FiberCtx.Query("latitude", "-10000.0")
	longitude := ctx.FiberCtx.Query("longitude", "-10000.0")

	latitudeFloat, _ := strconv.ParseFloat(latitude, 64)
	longitudeFloat, _ := strconv.ParseFloat(longitude, 64)
	freezingTempFloat, _ := strconv.ParseFloat(freezingTemp, 64)
	req := FreezingForecastRequest{
		Latitude:     latitudeFloat,
		Longitude:    longitudeFloat,
		FreezingTemp: freezingTempFloat,
	}
	err := req.Validate()
	if err != nil {
		panic(config.BAD_REQUEST.PutDetail("reason", err.Error()))
	}
	return req
}

func (r FreezingForecastRequest) Validate() error {
	return validation.ValidateStruct(&r,
		validation.Field(&r.Latitude, validation.Required, validation.Min(-90.0), validation.Max(90.0)),
		validation.Field(&r.Longitude, validation.Required, validation.Min(-180.0), validation.Max(180.0)),
		validation.Field(&r.FreezingTemp, validation.Required, validation.Min(0.0), validation.Max(40.0)),
	)
}

type FreezingForecastResponse struct {
	LastFreeze  []time.Time `json:"last_freeze"`
	FirstFreeze []time.Time `json:"first_freeze"`
}
