package models

import (
	"strconv"
	"time"

	validation "github.com/go-ozzo/ozzo-validation"

	"github.com/tgs266/dawn-go-common/common"
	"gitlab.cs.umd.edu/dawn/go-backend/dawn-gdd/config"
	"gitlab.cs.umd.edu/dawn/go-backend/dawn-gdd/persistence/entities"
)

type CultivarRequest struct {
	Name string `json:"name"`
}

type Cultivar struct {
	VarName string  `json:"var_name"`
	VarNum  string  `json:"var_num"`
	EcoNum  string  `json:"eco_num"`
	P1      float64 `json:"p1"`
	P2      float64 `json:"p2"`
	P5      float64 `json:"p5"`
	G2      float64 `json:"g2"`
	G3      float64 `json:"g3"`
	PHINT   float64 `json:"phint"`
}

type CultivarResponse struct {
	Cultivars []Cultivar `json:"cultivars"`
}

func CreateCultivar(c entities.Cultivar) Cultivar {
	return Cultivar{
		VarName: c.VarName,
		VarNum:  c.VarNum,
		EcoNum:  c.EcoNum,
		P1:      c.P1,
		P2:      c.P2,
		P5:      c.P5,
		G2:      c.P2,
		G3:      c.G3,
		PHINT:   c.PHINT,
	}
}

type CornMaturityRequest struct {
	Latitude  float64   `json:"latitude"`
	Longitude float64   `json:"longitude"`
	PlantDate time.Time `json:"plant_date"`
	VarName   string    `json:"var_name"`
	VarNum    string    `json:"var_num"`
	EcoNum    string    `json:"eco_num"`
	P1        float64   `json:"p1"`
	P2        float64   `json:"p2"`
	P5        float64   `json:"p5"`
	G2        float64   `json:"g2"`
	G3        float64   `json:"g3"`
	PHINT     float64   `json:"phint"`
}

type CornRMMaturityRequest struct {
	Latitude         float64   `json:"latitude"`
	Longitude        float64   `json:"longitude"`
	PlantDate        time.Time `json:"plant_date"`
	RelativeMaturity int       `json:"relative_maturity"`
}

type CornMaturityResponse struct {
	HarvestDate time.Time `json:"harvest_date"`
}

func (r CornMaturityRequest) Validate() error {

	return validation.ValidateStruct(&r,
		validation.Field(&r.Latitude, validation.Required, validation.Min(-90.0), validation.Max(90.0)),
		validation.Field(&r.Longitude, validation.Required, validation.Min(-180.0), validation.Max(180.0)),
	)
}

func (r CornRMMaturityRequest) Validate() error {

	return validation.ValidateStruct(&r,
		validation.Field(&r.Latitude, validation.Required, validation.Min(-90.0), validation.Max(90.0)),
		validation.Field(&r.Longitude, validation.Required, validation.Min(-180.0), validation.Max(180.0)),
	)
}

func BuildCornMaturityRequest(ctx common.DawnCtx) CornMaturityRequest {

	latitude, _ := strconv.ParseFloat(ctx.FiberCtx.Query("latitude", "-100000.0"), 64)
	longitude, _ := strconv.ParseFloat(ctx.FiberCtx.Query("longitude", "-100000.0"), 64)

	varName := ctx.FiberCtx.Query("var_name", "")
	varNum := ctx.FiberCtx.Query("var_num", "")
	ecoNum := ctx.FiberCtx.Query("eco_num", "")
	plant_month, _ := strconv.Atoi(ctx.FiberCtx.Query("plant_month", "0"))
	plant_day, _ := strconv.Atoi(ctx.FiberCtx.Query("plant_day", "0"))

	plant_date := time.Date(time.Now().Year(), time.Month(plant_month), plant_day, 0, 0, 0, 0, time.UTC)

	p1, _ := strconv.ParseFloat(ctx.FiberCtx.Query("p1", "0"), 64)
	p2, _ := strconv.ParseFloat(ctx.FiberCtx.Query("p2", "0"), 64)
	p5, _ := strconv.ParseFloat(ctx.FiberCtx.Query("p5", "0"), 64)
	g2, _ := strconv.ParseFloat(ctx.FiberCtx.Query("g2", "0"), 64)
	g3, _ := strconv.ParseFloat(ctx.FiberCtx.Query("g3", "0"), 64)
	phint, _ := strconv.ParseFloat(ctx.FiberCtx.Query("phint", "0"), 64)
	rNew := CornMaturityRequest{
		Latitude:  latitude,
		Longitude: longitude,
		VarName:   varName,
		VarNum:    varNum,
		EcoNum:    ecoNum,
		P1:        p1,
		P2:        p2,
		P5:        p5,
		G2:        g2,
		G3:        g3,
		PHINT:     phint,
		PlantDate: plant_date,
	}

	if rNew.Validate() != nil {
		panic(config.BAD_REQUEST)
	}
	return rNew

}

func BuildCornRMMaturityRequest(ctx common.DawnCtx) CornRMMaturityRequest {

	latitude, _ := strconv.ParseFloat(ctx.FiberCtx.Query("latitude", "-100000.0"), 64)
	longitude, _ := strconv.ParseFloat(ctx.FiberCtx.Query("longitude", "-100000.0"), 64)
	rm, _ := strconv.Atoi(ctx.FiberCtx.Query("relative_maturity", "0"))

	plant_month, _ := strconv.Atoi(ctx.FiberCtx.Query("plant_month", "0"))
	plant_day, _ := strconv.Atoi(ctx.FiberCtx.Query("plant_day", "0"))

	plant_date := time.Date(time.Now().Year(), time.Month(plant_month), plant_day, 0, 0, 0, 0, time.UTC)

	rNew := CornRMMaturityRequest{
		Latitude:         latitude,
		Longitude:        longitude,
		PlantDate:        plant_date,
		RelativeMaturity: rm,
	}

	if err := rNew.Validate(); err != nil {
		panic(config.BAD_REQUEST.PutDetail("reason", err.Error()))
	}
	return rNew

}
