package services

import (
	"testing"
	"time"

	DawnTest "github.com/tgs266/dawn-go-common/testing"

	"github.com/gofiber/fiber/v2"
	"github.com/stretchr/testify/assert"
	"github.com/tgs266/dawn-go-common/common"
	"gitlab.cs.umd.edu/dawn/go-backend/dawn-gdd/models"
	"gitlab.cs.umd.edu/dawn/go-backend/dawn-gdd/persistence"
	"gitlab.cs.umd.edu/dawn/go-backend/dawn-gdd/persistence/entities"
)

var CSVRequest = models.CSVRequest{
	Analog:         true,
	Cfs:            true,
	CfsLower:       true,
	CfsUpper:       true,
	Comparison:     true,
	FirstFreezing:  true,
	Gefs:           true,
	LastFreezing:   true,
	Maximum:        true,
	Minimum:        true,
	Normals:        true,
	Primary:        true,
	ComparisonYear: 2019,
	Product:        "soybean",
	Range:          95.0,
	Temperature:    28,
	Year:           2020,
	Latitude:       12.0,
	Longitude:      12.0,
}

func mockGetGddValues(c common.DawnCtx, request models.GddRequest) models.GddResponse {
	return models.GddResponse{
		Product:          request.Product,
		ClosestLatitude:  request.Latitude,
		ClosestLongitude: request.Longitude,
		GddValues:        []float64{1.0, 2.0, 3.0, 4.0},
		LastDate:         time.Date(2021, time.January, 4, 0, 0, 0, 0, time.Now().Location()),
	}
}

func mockGetCfsGddValues(request models.GddRequest) models.CfsGddResponse {
	return models.CfsGddResponse{
		Product:          request.Product,
		ClosestLatitude:  request.Latitude,
		ClosestLongitude: request.Longitude,
		GddValues:        []float64{1.0, 2.0, 3.0, 4.0, 5.0, 6.0, 7.0, 8.0, 9.0, 10.0, 11.0, 12.0, 13.0},
		UpperBound:       []float64{1.0, 2.0, 3.0, 4.0, 5.0, 6.0, 7.0, 8.0, 9.0, 10.0, 11.0, 12.0, 13.0},
		LowerBound:       []float64{1.0, 2.0, 3.0, 4.0, 5.0, 6.0, 7.0, 8.0, 9.0, 10.0, 11.0, 12.0, 13.0},
		FirstDate:        time.Date(2021, time.January, 3, 0, 0, 0, 0, time.UTC),
	}
}

func mockGetGefsGddValues(request models.GddRequest) models.GefsGddResponse {
	return models.GefsGddResponse{
		Product:          request.Product,
		ClosestLatitude:  request.Latitude,
		ClosestLongitude: request.Longitude,
		GddValues:        []float64{1.0, 2.0, 3.0, 4.0, 5.0, 6.0, 7.0, 8.0, 9.0, 10.0, 11.0, 12.0, 13.0},
		UpperBound:       []float64{1.0, 2.0, 3.0, 4.0, 5.0, 6.0, 7.0, 8.0, 9.0, 10.0, 11.0, 12.0, 13.0},
		LowerBound:       []float64{1.0, 2.0, 3.0, 4.0, 5.0, 6.0, 7.0, 8.0, 9.0, 10.0, 11.0, 12.0, 13.0},
		FirstDate:        time.Date(2021, time.January, 3, 0, 0, 0, 0, time.UTC),
		LastDate:         time.Date(2021, time.January, 13, 0, 0, 0, 0, time.UTC),
	}
}

func mockGetNormalValues(request models.GddRequest) models.GddResponse {
	return models.GddResponse{
		Product:          request.Product,
		ClosestLatitude:  request.Latitude,
		ClosestLongitude: request.Longitude,
		GddValues:        []float64{1.0, 2.0, 3.0, 4.0},
		LastDate:         time.Date(2021, time.January, 4, 0, 0, 0, 0, time.Now().Location()),
	}
}

func mockFindAnalogYear(location entities.Location) models.AnalogResponse {
	return models.AnalogResponse{
		ClosestLatitude:  4.0,
		ClosestLongitude: 4.0,
		AnalogYear:       2001,
	}
}

func mockGetConfidenceInterval(request models.ConfidenceIntervalRequest) models.ConfidenceIntervalResposne {
	return models.ConfidenceIntervalResposne{
		ClosestLatitude:  request.Latitude,
		ClosestLongitude: request.Longitude,
		LowerBound:       []float64{1.0, 1.0, 2.0, 2.0},
		UpperBound:       []float64{5.0, 5.0, 10.0, 10.0},
	}
}

func mockGetFreezingDate(request models.FreezingDateRequest) models.FreezingDateResponse {
	firstDateCounts := make(map[string]*models.SingleFreezingDate)
	firstDateCounts["01-04"] = &models.SingleFreezingDate{
		Count: 4,
		Years: []int{2020, 2014, 2013, 2011},
	}
	firstDateCounts["03-04"] = &models.SingleFreezingDate{
		Count: 3,
		Years: []int{1988, 2005, 1985},
	}

	lastDateCounts := make(map[string]*models.SingleFreezingDate)
	lastDateCounts["10-04"] = &models.SingleFreezingDate{
		Count: 4,
		Years: []int{2020, 2014, 2013, 2011},
	}
	lastDateCounts["10-04"] = &models.SingleFreezingDate{
		Count: 3,
		Years: []int{1988, 2005, 1985},
	}
	return models.FreezingDateResponse{
		ClosestLatitude:  request.Latitude,
		ClosestLongitude: request.Longitude,
		FirstDateCounts:  firstDateCounts,
		LastDateCounts:   lastDateCounts,
	}
}

func TestFillKeys(t *testing.T) {
	keys := fillKeys(CSVRequest)
	assert.Equal(t, "Date", keys[0])
	assert.Equal(t, "Analog Year GDD", keys[1])
	assert.Equal(t, "Primary GDD (2020)", keys[12])
}

func TestParseDate(t *testing.T) {
	year := 2021

	date := "01-05"
	nd, _ := time.Parse("2006-Jan-02", "2021-Jan-05")
	assert.Equal(t, nd, parseDate(date, year))

	date = "02-05"
	nd, _ = time.Parse("2006-Jan-02", "2021-Feb-05")
	assert.Equal(t, nd, parseDate(date, year))

	date = "03-05"
	nd, _ = time.Parse("2006-Jan-02", "2021-Mar-05")
	assert.Equal(t, nd, parseDate(date, year))

	date = "04-05"
	nd, _ = time.Parse("2006-Jan-02", "2021-Apr-05")
	assert.Equal(t, nd, parseDate(date, year))

	date = "05-05"
	nd, _ = time.Parse("2006-Jan-02", "2021-May-05")
	assert.Equal(t, nd, parseDate(date, year))

	date = "06-05"
	nd, _ = time.Parse("2006-Jan-02", "2021-Jun-05")
	assert.Equal(t, nd, parseDate(date, year))

	date = "07-05"
	nd, _ = time.Parse("2006-Jan-02", "2021-Jul-05")
	assert.Equal(t, nd, parseDate(date, year))

	date = "08-05"
	nd, _ = time.Parse("2006-Jan-02", "2021-Aug-05")
	assert.Equal(t, nd, parseDate(date, year))

	date = "09-05"
	nd, _ = time.Parse("2006-Jan-02", "2021-Sep-05")
	assert.Equal(t, nd, parseDate(date, year))

	date = "10-05"
	nd, _ = time.Parse("2006-Jan-02", "2021-Oct-05")
	assert.Equal(t, nd, parseDate(date, year))

	date = "11-05"
	nd, _ = time.Parse("2006-Jan-02", "2021-Nov-05")
	assert.Equal(t, nd, parseDate(date, year))

	date = "12-05"
	nd, _ = time.Parse("2006-Jan-02", "2021-Dec-05")
	assert.Equal(t, nd, parseDate(date, year))
}

func TestGetPrimary(t *testing.T) {
	mock := DawnTest.CreateMock(GetGddValues, mockGetGddValues)
	defer mock.Unpatch()

	dates := fillDates()

	response := getPrimary(common.DawnCtx{}, CSVRequest, dates)
	assert.Equal(t, 1.0, response[0])
}

func TestGetNormals(t *testing.T) {
	mock := DawnTest.CreateMock(GetNormalValues, mockGetNormalValues)
	defer mock.Unpatch()

	dates := fillDates()

	response := getNormals(CSVRequest, dates)
	assert.Equal(t, 1.0, response[0])
}

func TestGetAnalogYear(t *testing.T) {
	mock := DawnTest.CreateMock(persistence.FindAnalogYear, mockFindAnalogYear)
	mock.AddMock(GetGddValues, mockGetGddValues)

	defer mock.Unpatch()

	dates := fillDates()

	response := getAnalogYear(&fiber.Ctx{}, CSVRequest, dates)
	assert.Equal(t, 1.0, response[0])
}

func TestGetFreezingDates(t *testing.T) {
	mock := DawnTest.CreateMock(GetFreezingDate, mockGetFreezingDate)

	defer mock.Unpatch()

	dates := fillDates()

	response := getFreezingDates(CSVRequest, dates)
	assert.Equal(t, 4, response[0][3])
}

func TestGetComparisonYear(t *testing.T) {
	mock := DawnTest.CreateMock(GetGddValues, mockGetGddValues)

	defer mock.Unpatch()

	dates := fillDates()

	response := getComparisonYear(common.DawnCtx{}, CSVRequest, dates)
	assert.Equal(t, 1.0, response[0])
}

func TestGetConfidenceInterval(t *testing.T) {
	mock := DawnTest.CreateMock(GetConfidenceInterval, mockGetConfidenceInterval)

	defer mock.Unpatch()

	dates := fillDates()

	response := getConfidenceInterval(CSVRequest, dates)
	assert.Equal(t, 1.0, response[0][0])

}

func TestGetCfsData(t *testing.T) {
	mock := DawnTest.CreateMock(GetCfsGddValues, mockGetCfsGddValues)

	defer mock.Unpatch()

	dates := fillDates()

	response := getCfsData(CSVRequest, dates)
	assert.Equal(t, 1.0, response[0][2])

}

func TestGetGefsGdd(t *testing.T) {
	mock := DawnTest.CreateMock(GetGefsGddValues, mockGetGefsGddValues)

	defer mock.Unpatch()

	dates := fillDates()

	response := getGefsData(CSVRequest, dates)
	assert.Equal(t, 1.0, response[2])
}

func TestPullData(t *testing.T) {
	mock := DawnTest.CreateMock(persistence.FindAnalogYear, mockFindAnalogYear)
	mock.AddMock(GetCfsGddValues, mockGetCfsGddValues)
	mock.AddMock(GetGddValues, mockGetGddValues)
	mock.AddMock(GetFreezingDate, mockGetFreezingDate)
	mock.AddMock(GetGefsGddValues, mockGetGefsGddValues)
	mock.AddMock(GetConfidenceInterval, mockGetConfidenceInterval)
	mock.AddMock(GetNormalValues, mockGetNormalValues)

	defer mock.Unpatch()

	response := pullData(common.DawnCtx{}, CSVRequest)
	assert.Equal(t, 1.0, response.Primary[0])
}
