package utils

import (
	"testing"

	"gitlab.cs.umd.edu/dawn/go-backend/dawn-gdd/models/enums"

	"github.com/stretchr/testify/assert"
)

var minTempsArr = [...]float64{55.5, 62.4, 22.5, 96.5}
var maxTempsArr = [...]float64{78.4, 92.5, 55.8, 35.2}

var normalBaseArr = [...]float64{50, 60, 70, 80}

func Test_CalculateSingleGdd(t *testing.T) {
	minTemp := minTempsArr[0]
	maxTemp := maxTempsArr[0]

	expectedGdd := (minTemp + maxTemp) / 2
	expectedGdd -= enums.ProductType.SOYBEAN.BaseTemp

	assert.Equal(t, expectedGdd, CalculateSingleGdd(minTemp, maxTemp, enums.ProductType.SOYBEAN))
}

func Test_CalculateSingleGdd_Corn(t *testing.T) {
	minTemp := minTempsArr[0]
	maxTemp := maxTempsArr[0]

	expectedGdd := (minTemp + maxTemp) / 2
	expectedGdd -= enums.ProductType.CORN.BaseTemp

	assert.Equal(t, expectedGdd, CalculateSingleGdd(minTemp, maxTemp, enums.ProductType.CORN))
}

func Test_CalculateGddValues(t *testing.T) {
	var expectedGdd []float64

	for i := 0; i < 4; i++ {
		val := (minTempsArr[i] + maxTempsArr[i]) / 2
		val -= enums.ProductType.OAT.BaseTemp
		expectedGdd = append(expectedGdd, ClipMinFloat(val, 0))
	}

	assert.Equal(t, expectedGdd, CalculateGddValues(minTempsArr[:], maxTempsArr[:], enums.ProductType.OAT, false))
}

func Test_CalculateGddValues_Corn(t *testing.T) {
	var expectedGdd []float64

	for i := 0; i < 4; i++ {
		val := (ClipFloat(minTempsArr[i], 50, 86) + ClipFloat(maxTempsArr[i], 50, 86)) / 2
		val -= enums.ProductType.CORN.BaseTemp
		expectedGdd = append(expectedGdd, val)
	}

	assert.Equal(t, expectedGdd, CalculateGddValues(minTempsArr[:], maxTempsArr[:], enums.ProductType.CORN, false))
}

func Test_CalculateGddValues_Accumulated(t *testing.T) {
	var expectedGdd []float64
	for i := 0; i < 4; i++ {
		val := (minTempsArr[i] + maxTempsArr[i]) / 2
		val -= enums.ProductType.WHEAT.BaseTemp
		val = ClipMinFloat(val, 0)
		if i > 0 {
			val += expectedGdd[len(expectedGdd)-1]
		}
		expectedGdd = append(expectedGdd, val)
	}

	assert.Equal(t, expectedGdd, CalculateGddValues(minTempsArr[:], maxTempsArr[:], enums.ProductType.WHEAT, true))
}

func Test_CalculateNormalGddValues(t *testing.T) {
	var expectedGdd []float64
	for i := 0; i < 4; i++ {
		val := normalBaseArr[i] - enums.ProductType.PEANUT.BaseTemp
		val = ClipMinFloat(val, 0)
		expectedGdd = append(expectedGdd, val)
	}

	assert.Equal(t, expectedGdd, CalculateNormalGddValues(normalBaseArr[:], enums.ProductType.PEANUT, false))
}

func Test_CalculateNormalGddValues_Accumulated(t *testing.T) {
	var expectedGdd []float64
	for i := 0; i < 4; i++ {
		val := normalBaseArr[i] - enums.ProductType.PEANUT.BaseTemp
		val = ClipMinFloat(val, 0)
		if i > 0 {
			val += expectedGdd[len(expectedGdd)-1]
		}
		expectedGdd = append(expectedGdd, val)
	}

	assert.Equal(t, expectedGdd, CalculateNormalGddValues(normalBaseArr[:], enums.ProductType.PEANUT, true))
}
