package controllers

import (
	"github.com/gofiber/fiber/v2"
	"github.com/tgs266/dawn-go-common/common"
	"gitlab.cs.umd.edu/dawn/go-backend/dawn-gdd/models"
	"gitlab.cs.umd.edu/dawn/go-backend/dawn-gdd/services"
)

type MaturityRequest struct {
	PlantDate int       `json:"plantdate"`
	Tmins     []float32 `json:"tmins"`
	Tmaxs     []float32 `json:"tmaxs"`
	Srads     []float32 `json:"srads"`
	Rains     []float32 `json:"rains"`
	Units     string    `json:"units"`
}

type ResponseTest struct {
	Date int `json:"date"`
}

// GetCultivars godoc
// @Summary Get all generic corn cultivars
// @Tags Maturity
// @Description Get all generic corn cultivars
// @Accept  json
// @Produce  json
// @Success 200 {object} models.CultivarResponse
// @Failure 400 {object} common.StandardError
// @Param name query string false "Name of cultivar to get"
// @Router /gdd/maturity/corn/cultivars [get]
func GetCultivars(c *fiber.Ctx) error {
	ctx := common.DawnCtx{FiberCtx: c}
	request := models.CultivarRequest{
		Name: c.Query("name", ""),
	}
	return c.Status(fiber.StatusOK).JSON(services.GetCultivars(ctx, request))
}

// GetMaturity godoc
// @Summary Get expected maturity date
// @Tags Maturity
// @Description Get expected maturity date
// @Accept  json
// @Produce  json
// @Success 200 {object} models.CornMaturityResponse
// @Failure 400 {object} common.StandardError
// @Param latitude query number true "Latitude to search for"
// @Param longitude query number true "Longitude to search for"
// @Param plant_month query number true "Platning month"
// @Param plant_date query number true "Platning date"
// @Param var_name query string true "Name of cultivar"
// @Param var_num query string true "ID# of cultivar"
// @Param eco_num query string true "Eco# of cultivar"
// @Param p1 query number true "P1 value of cultivar"
// @Param p2 query number true "P2 value of cultivar"
// @Param p5 query number true "P5 value of cultivar"
// @Param g2 query number true "G2 value of cultivar"
// @Param g3 query number true "G3 value of cultivar"
// @Param phint query number true "PHINT value of cultivar"
// @Router /gdd/maturity/corn [get]
func GetMaturity(c *fiber.Ctx) error {
	ctx := common.DawnCtx{FiberCtx: c}
	request := models.BuildCornMaturityRequest(ctx)
	return c.Status(fiber.StatusOK).JSON(services.CalculateMaturity(ctx, request))
}

// GetMaturity godoc
// @Summary Get expected maturity date
// @Tags Maturity
// @Description Get expected maturity date
// @Accept  json
// @Produce  json
// @Success 200 {object} models.CornMaturityResponse
// @Failure 400 {object} common.StandardError
// @Param latitude query number true "Latitude to search for"
// @Param longitude query number true "Longitude to search for"
// @Param plant_month query number true "Platning month"
// @Param plant_date query number true "Platning date"
// @Param relative_maturity query number true "RM of seed"
// @Router /gdd/maturity/corn/relative [get]
func GetRMToMaturity(c *fiber.Ctx) error {
	ctx := common.DawnCtx{FiberCtx: c}
	request := models.BuildCornRMMaturityRequest(ctx)
	return c.Status(fiber.StatusOK).JSON(services.CalculateRMMaturity(ctx, request))
}
