package models

import (
	"strconv"
	"time"

	"gitlab.cs.umd.edu/dawn/go-backend/dawn-gdd/config"

	validation "github.com/go-ozzo/ozzo-validation"
	"github.com/gofiber/fiber/v2"
)

type CSVRequest struct {
	Analog         bool    `json:"analog"`
	Cfs            bool    `json:"cfs"`
	CfsLower       bool    `json:"cfs_lower"`
	CfsUpper       bool    `json:"cfs_upper"`
	Comparison     bool    `json:"comparison"`
	FirstFreezing  bool    `json:"first_freezing"`
	Gefs           bool    `json:"gefs"`
	LastFreezing   bool    `json:"last_freezing"`
	Maximum        bool    `json:"maximum"`
	Minimum        bool    `json:"minimum"`
	Normals        bool    `json:"normals"`
	Primary        bool    `json:"primary"`
	ComparisonYear int     `json:"comparison_year"`
	Product        string  `json:"product"`
	Range          float64 `json:"range"`
	Temperature    int     `json:"temperature"`
	Year           int     `json:"year"`
	Latitude       float64 `json:"latitude"`
	Longitude      float64 `json:"longitude"`
}

func (r CSVRequest) Validate() error {
	return validation.ValidateStruct(&r,
		validation.Field(&r.Year, validation.Required, validation.Min(1981), validation.Max(time.Now().Year())),
		validation.Field(&r.Product, validation.Required, validation.Length(1, 100)),
		validation.Field(&r.Latitude, validation.Required, validation.Min(-90.0), validation.Max(90.0)),
		validation.Field(&r.Longitude, validation.Required, validation.Min(-180.0), validation.Max(180.0)),
	)
}

func parseBool(str string) bool {
	a, e := strconv.ParseBool(str)
	if e != nil {
		return false
	}
	return a
}

func atoi(str string) int {
	a, e := strconv.Atoi(str)
	if e != nil {
		return 0
	}
	return a
}

func (r CSVRequest) Build(c *fiber.Ctx) CSVRequest {

	latitude, _ := strconv.ParseFloat(c.Query("latitude", "-1000.0"), 64)
	longitude, _ := strconv.ParseFloat(c.Query("longitude", "-1000.0"), 64)
	_range, _ := strconv.ParseFloat(c.Query("range", "-1000.0"), 64)

	newRequest := CSVRequest{
		Analog:        parseBool(c.Query("analog")),
		Cfs:           parseBool(c.Query("cfs")),
		CfsLower:      parseBool(c.Query("cfs_lower")),
		CfsUpper:      parseBool(c.Query("cfs_upper")),
		Comparison:    parseBool(c.Query("comparison")),
		FirstFreezing: parseBool(c.Query("first_freezing")),
		Gefs:          parseBool(c.Query("gefs")),
		LastFreezing:  parseBool(c.Query("last_freezing")),
		Maximum:       parseBool(c.Query("maximum")),
		Minimum:       parseBool(c.Query("minimum")),
		Normals:       parseBool(c.Query("normals")),
		Primary:       parseBool(c.Query("primary")),

		ComparisonYear: atoi(c.Query("comparison_year", "0")),
		Product:        c.Query("product", "NONE"),
		Range:          _range,
		Temperature:    atoi(c.Query("temperature", "0")),
		Year:           atoi(c.Query("year", "0")),
		Latitude:       latitude,
		Longitude:      longitude,
	}

	if e := newRequest.Validate(); e != nil {
		panic(config.BAD_REQUEST.AddLogDetails(e.Error()))
	}

	return newRequest
}
