package models

import (
	"strconv"
	"time"

	"gitlab.cs.umd.edu/dawn/go-backend/dawn-gdd/config"
	"gitlab.cs.umd.edu/dawn/go-backend/dawn-gdd/persistence/entities"

	validation "github.com/go-ozzo/ozzo-validation"
	"github.com/gofiber/fiber/v2"
)

type DateCount struct {
	Date  time.Time `json:"date"`
	Count int       `json:"count"`
}

type SingleFreezingDate struct {
	Count int   `json:"count"`
	Years []int `json:"years"`
}

func (s *SingleFreezingDate) IncreaseCount() {
	s.Count = s.Count + 1
}

func (s *SingleFreezingDate) AddYear(year int) {
	s.Years = append(s.Years, year)
}

type FreezingDateResponse struct {
	ClosestLatitude  float64                        `json:"closest_latitude"`
	ClosestLongitude float64                        `json:"closest_longitude"`
	FirstDateCounts  map[string]*SingleFreezingDate `json:"first_date_counts"`
	LastDateCounts   map[string]*SingleFreezingDate `json:"last_date_counts"`
}

type FreezingDateRequest struct {
	Latitude     float64 `json:"latitude"`
	Longitude    float64 `json:"longitude"`
	FreezingTemp int     `json:"freezing_temp"`
}

func (r FreezingDateRequest) Validate() error {
	return validation.ValidateStruct(&r,
		validation.Field(&r.Latitude, validation.Required, validation.Min(-90.0), validation.Max(90.0)),
		validation.Field(&r.Longitude, validation.Required, validation.Min(-180.0), validation.Max(180.0)),
		validation.Field(&r.FreezingTemp, validation.Required, validation.Min(25), validation.Max(32)),
	)
}

func (r FreezingDateRequest) Build(c *fiber.Ctx) FreezingDateRequest {
	latitude, _ := strconv.ParseFloat(c.Query("latitude", "-10000"), 64)
	longitude, _ := strconv.ParseFloat(c.Query("longitude", "-10000"), 64)
	freezingTemp, _ := strconv.Atoi(c.Query("freezing_temp", "0"))

	rNew := FreezingDateRequest{
		Latitude:     latitude,
		Longitude:    longitude,
		FreezingTemp: freezingTemp,
	}

	if rNew.Validate() != nil {
		panic(config.BAD_REQUEST)
	}

	return rNew
}

func (r FreezingDateRequest) BuildLocation() entities.Location {
	l := entities.Location{
		Type:        "Point",
		Coordinates: []float64{r.Longitude, r.Latitude},
	}
	return l
}
