package main

import (
	"dawn-weather/common"
	"dawn-weather/config"
	"dawn-weather/controllers"
	"dawn-weather/persistence"
	"strconv"

	"github.com/ansrivas/fiberprometheus/v2"
	swagger "github.com/arsmn/fiber-swagger/v2"
	"github.com/gofiber/fiber/v2"
	"github.com/gofiber/fiber/v2/middleware/cors"
	"github.com/gofiber/fiber/v2/middleware/logger"
	"github.com/gofiber/fiber/v2/middleware/recover"
	"github.com/gofiber/fiber/v2/middleware/requestid"
	"github.com/gofiber/fiber/v2/utils"
	"github.com/spf13/viper"

	_ "dawn-weather/docs"
)

func registerRoutes(app *fiber.App) {
	api := app.Group(viper.GetString("server.context-path"))
	controllers.GddRoutes(api)
}

func registerSwagger(app *fiber.App) {
	if viper.GetBool("app.swagger") {
		app.Get(viper.GetString("server.context-path")+"/gdd-swagger/*", swagger.Handler)
		app.Get(viper.GetString("server.context-path")+"/gdd-swagger/*", swagger.New())
	}
}

func registerCors(app *fiber.App) {
	app.Use(cors.New(cors.Config{
		AllowOrigins: "http://localhost:3000, http://localhost:4200, http://localhost:5000, http://localhost:12321",
		AllowHeaders: "Origin, Content-Type, Accept",
	}))
}

func registerLogging(app *fiber.App) {
	app.Use(requestid.New(requestid.Config{
		Next:   nil,
		Header: fiber.HeaderXRequestID,
		Generator: func() string {
			return utils.UUIDv4()
		},
		ContextKey: "requestId",
	}))

	app.Use(logger.New())
}

func registerPrometheus(app *fiber.App) {
	prometheus := fiberprometheus.New(viper.GetString("app.name"))
	prometheus.RegisterAt(app, viper.GetString("server.context-path")+"/metrics")
	app.Use(prometheus.Middleware)
}

func createFiberConfig() fiber.Config {
	return fiber.Config{
		ErrorHandler: func(ctx *fiber.Ctx, err error) error {

			code := fiber.StatusInternalServerError
			message := common.StandardError{Source: viper.GetString("app.name"), ErrorCode: "INTERNAL_SERVER",
				Description: "Internal Server Error Occurred", Details: map[string]string{"RequestId": ""}}

			if e, ok := err.(*common.DawnError); ok {
				code = e.Code
				message = err.(*common.DawnError).BuildStandardError(ctx)
			} else {
				err = common.Build(err)
			}

			logMessage := common.BuildMessage(ctx)
			logMessage.Error = err.(*common.DawnError)
			logMessage.StatusCode = strconv.Itoa(code)

			common.LogRequest(logMessage)

			if code == 500 {
				message = config.INTERNAL_SERVER_STANDARD_ERROR.BuildStandardError(ctx)
			}

			err = ctx.Status(code).JSON(message)

			return nil
		},
	}
}

func CreateFiberApp() *fiber.App {
	app := fiber.New(createFiberConfig())
	app.Use(recover.New())
	registerCors(app)
	registerSwagger(app)
	registerPrometheus(app)
	registerLogging(app)
	registerRoutes(app)
	return app
}

// @title Dawn GDD Service
// @version 1.0
// @description All operations for GDD/Freezing Date data
// @contact.name API Support
// @contact.email tgsiegel@umd.edu
// @host localhost:8080
// @BasePath /
func main() {
	config.GetConfig()
	persistence.CreateDBSession()
	app := CreateFiberApp()

	err := app.Listen(":" + viper.GetString("server.port"))

	if err != nil {
		panic(err)
	}
}
