package models

import (
	"dawn-weather/config"
	"dawn-weather/models/enums"
	"dawn-weather/persistence/entities"
	"errors"
	"strconv"

	validation "github.com/go-ozzo/ozzo-validation"
	"github.com/gofiber/fiber/v2"
)

type ConfidenceIntervalRequest struct {
	Product   enums.Product `json:"product"`
	Latitude  float64       `json:"latitude"`
	Longitude float64       `json:"longitude"`
	Interval  float64       `json:"interval"`
}

type ConfidenceIntervalResposne struct {
	LowerBound       []float64 `json:"lower_bound,omitempty"`
	UpperBound       []float64 `json:"upper_bound,omitempty"`
	ClosestLatitude  float64   `json:"closest_latitude,omitempty"`
	ClosestLongitude float64   `json:"closest_longitude,omitempty"`
}

var IntervalConvert = map[float64]float64{
	80:   1.282,
	85:   1.440,
	90:   1.645,
	95:   1.960,
	99:   2.576,
	99.5: 2.807,
	99.9: 3.291,
}

func validateInterval(value interface{}) error {
	for k := range IntervalConvert {
		if k == value {
			return nil
		}
	}
	return errors.New("Interval not in allowed intervals")
}

func (r ConfidenceIntervalRequest) Validate() error {

	return validation.ValidateStruct(&r,
		validation.Field(&r.Latitude, validation.Required, validation.Min(-90.0), validation.Max(90.0)),
		validation.Field(&r.Longitude, validation.Required, validation.Min(-180.0), validation.Max(180.0)),
		validation.Field(&r.Interval, validation.Required, validation.By(validateInterval)),
	)
}

func (r ConfidenceIntervalRequest) Build(c *fiber.Ctx) ConfidenceIntervalRequest {
	product := c.Query("product", "NONE")
	latitude, _ := strconv.ParseFloat(c.Query("latitude", "-100000.0"), 64)
	longitude, _ := strconv.ParseFloat(c.Query("longitude", "-100000.0"), 64)
	interval, _ := strconv.ParseFloat(c.Query("interval", "-100000.0"), 64)

	rNew := ConfidenceIntervalRequest{
		Product:   enums.GetProductFromString(product),
		Latitude:  latitude,
		Longitude: longitude,
		Interval:  interval,
	}

	if rNew.Validate() != nil {
		panic(config.BAD_REQUEST)
	}

	return rNew
}

func (r ConfidenceIntervalRequest) BuildLocation() entities.Location {
	l := entities.Location{
		Type:        "Point",
		Coordinates: []float64{r.Longitude, r.Latitude},
	}
	return l
}
