package models

import (
	"dawn-weather/config"
	"strconv"
	"time"

	validation "github.com/go-ozzo/ozzo-validation"
	"github.com/gofiber/fiber/v2"
)

type SeedListRequest struct {
	Product string `json:"product"`
}

type SeedListResponse struct {
	Seeds []string `json:"seeds"`
}

type CornMaturityRequest struct {
	Latitude  float64 `json:"latitude"`
	Longitude float64 `json:"longitude"`
	Seed      string  `json:"seed"`
	Month     int     `json:"month"`
	Date      int     `json:"date"`
}

type CornMaturityResponse struct {
	Date             time.Time `json:"date"`
	GDD              float64   `json:"gdd"`
	ClosestGDD       float64   `json:"closest_gdd"`
	ClosestLatitude  float64   `json:"closest_latitude"`
	ClosestLongitude float64   `json:"closest_longitude"`
}

func (r CornMaturityRequest) Validate() error {
	return validation.ValidateStruct(&r,
		validation.Field(&r.Latitude, validation.Required, validation.Min(-90.0), validation.Max(90.0)),
		validation.Field(&r.Longitude, validation.Required, validation.Min(-180.0), validation.Max(180.0)),
		validation.Field(&r.Month, validation.Required, validation.Min(1), validation.Max(12)),
		validation.Field(&r.Date, validation.Required, validation.Min(1), validation.Max(31)),
	)
}

func BuildCornMaturityRequest(c *fiber.Ctx) CornMaturityRequest {

	lat, _ := strconv.ParseFloat(c.Query("latitude", "-10000.0"), 64)
	lon, _ := strconv.ParseFloat(c.Query("longitude", "-10000.0"), 64)
	month, _ := strconv.Atoi(c.Query("month", "0"))
	date, _ := strconv.Atoi(c.Query("date", "0"))
	seed := c.Query("seed", "-10000.0")

	newRequest := CornMaturityRequest{
		Latitude:  lat,
		Longitude: lon,
		Seed:      seed,
		Month:     month,
		Date:      date,
	}

	if e := newRequest.Validate(); e != nil {
		panic(config.BAD_REQUEST.AddLogDetails(e.Error()))
	}

	return newRequest
}
