package services

import (
	"dawn-weather/common"
	"dawn-weather/config"
	"dawn-weather/models"
	"dawn-weather/models/enums"
	"dawn-weather/persistence"
	"dawn-weather/persistence/entities"
	"encoding/csv"
	"fmt"
	"io/ioutil"
	"log"
	"math"
	"os"
	"strconv"
	"strings"
	"time"

	"github.com/gofiber/fiber/v2"
	"github.com/google/uuid"
)

type CSVData struct {
	Date          []time.Time
	Analog        []float64
	Cfs           []float64
	CfsLower      []float64
	CfsUpper      []float64
	Comparison    []float64
	FirstFreezing []int
	LastFreezing  []int
	GEFS          []float64
	Maximum       []float64
	Minimum       []float64
	Normals       []float64
	Primary       []float64
}

func fillKeys(request models.CSVRequest) []string {
	keys := []string{}

	keys = append(keys, "Date")
	if request.Analog {
		keys = append(keys, "Analog Year GDD")
	}
	if request.Cfs || request.CfsLower || request.CfsUpper {
		keys = append(keys, "CFS GDD")
		keys = append(keys, "CFS Lower Boundary GDD")
		keys = append(keys, "CFS Upper Boundary GDD")
	}
	if request.Comparison {
		keys = append(keys, "Comparison Year GDD ("+strconv.Itoa(request.ComparisonYear)+")")
	}
	if request.FirstFreezing || request.LastFreezing {
		keys = append(keys, "First Freezing Date Counts ("+strconv.Itoa(request.Temperature)+" Degrees F)")
		keys = append(keys, "Last Freezing Date Counts ("+strconv.Itoa(request.Temperature)+" Degrees F)")
	}
	if request.Gefs {
		keys = append(keys, "GEFS GDD")
	}
	if request.Maximum {
		keys = append(keys, "Confidence Interval Upper Boundary")
	}
	if request.Minimum {
		keys = append(keys, "Confidence Interval Lower Boundary")
	}
	if request.Normals {
		keys = append(keys, "Thirty Year Normal GDD")
	}
	if request.Primary {
		keys = append(keys, "Primary GDD ("+strconv.Itoa(request.Year)+")")
	}
	return keys
}

func timeToString(timeValue time.Time) string {
	day := timeValue.Day()
	month := int(timeValue.Month())
	timeString := fmt.Sprintf("%02d", month) + "-" + fmt.Sprintf("%02d", day)
	return timeString
}

func fillDates() []time.Time {
	dates := []time.Time{}
	date := time.Date(1981, time.January, 1, 0, 0, 0, 0, time.UTC)

	for i := 0; i < 365; i++ {
		dates = append(dates, date)
		date = date.Add(time.Duration(24) * time.Hour)
	}

	return dates
}

func getPrimary(c common.DawnCtx, request models.CSVRequest, dates []time.Time) []float64 {
	gddRequest := models.GddRequest{
		Year:       request.Year,
		Product:    request.Product,
		Latitude:   request.Latitude,
		Longitude:  request.Longitude,
		Accumulate: true,
	}

	gddValues := GetGddValues(c, gddRequest).GddValues
	for len(gddValues) < 365 {
		gddValues = append(gddValues, math.NaN())
	}
	return gddValues
}

func getNormals(request models.CSVRequest, dates []time.Time) []float64 {
	gddRequest := models.GddRequest{
		Year:       request.Year,
		Product:    request.Product,
		Latitude:   request.Latitude,
		Longitude:  request.Longitude,
		Accumulate: true,
	}

	gddValues := GetNormalValues(gddRequest).GddValues
	for len(gddValues) < 365 {
		gddValues = append(gddValues, math.NaN())
	}
	return gddValues
}

func getAnalogYear(c *fiber.Ctx, request models.CSVRequest, dates []time.Time) []float64 {

	location := entities.Location{
		Type:        "Point",
		Coordinates: []float64{request.Longitude, request.Latitude},
	}

	analogYear := persistence.FindAnalogYear(location).AnalogYear

	gddRequest := models.GddRequest{
		Year:       analogYear,
		Product:    request.Product,
		Latitude:   request.Latitude,
		Longitude:  request.Longitude,
		Accumulate: true,
	}

	gddValues := GetNormalValues(gddRequest).GddValues
	for len(gddValues) < 365 {
		gddValues = append(gddValues, math.NaN())
	}
	return gddValues
}

func _parseDate(date string) time.Time {
	const layout = "2006-Jan-02"
	newDate, e := time.Parse(layout, date)
	if e != nil {
		panic(config.DATE_PARSE_FAILURE.AddLogDetails(e.Error()))
	}
	return newDate
}

func parseDate(date string) time.Time {

	split := strings.Split(date, "-")
	day := split[1]
	switch split[0] {
	case "01":
		return _parseDate("1981-Jan-" + day)
	case "02":
		return _parseDate("1981-Feb-" + day)
	case "03":
		return _parseDate("1981-Mar-" + day)
	case "04":
		return _parseDate("1981-Apr-" + day)
	case "05":
		return _parseDate("1981-May-" + day)
	case "06":
		return _parseDate("1981-Jun-" + day)
	case "07":
		return _parseDate("1981-Jul-" + day)
	case "08":
		return _parseDate("1981-Aug-" + day)
	case "09":
		return _parseDate("1981-Sep-" + day)
	case "10":
		return _parseDate("1981-Oct-" + day)
	case "11":
		return _parseDate("1981-Nov-" + day)
	case "12":
		return _parseDate("1981-Dev-" + day)
	}
	panic(config.DATE_PARSE_FAILURE.AddLogDetails("Failed converting " + date + " to proper format."))
}

func getFreezingDates(request models.CSVRequest, dates []time.Time) [][]int {

	freezingRequest := models.FreezingDateRequest{
		Latitude:     request.Latitude,
		Longitude:    request.Longitude,
		FreezingTemp: request.Temperature,
	}

	response := GetFreezingDate(freezingRequest)

	firstFreezingDates := make(map[time.Time]int)
	lastFreezingDates := make(map[time.Time]int)
	for k, v := range response.LastDateCounts {
		date := parseDate(k)
		lastFreezingDates[date] = v
	}
	for k, v := range response.FirstDateCounts {
		date := parseDate(k)
		firstFreezingDates[date] = v
	}

	date := time.Date(1981, time.January, 1, 0, 0, 0, 0, time.UTC)
	lastFreezingValues := []int{}
	firstFreezingValues := []int{}
	for i := 0; i < 366; i++ {
		if val, ok := lastFreezingDates[date]; ok {
			lastFreezingValues = append(lastFreezingValues, val)
			firstFreezingValues = append(firstFreezingValues, int(math.NaN()))
		} else if val, ok := firstFreezingDates[date]; ok {
			firstFreezingValues = append(firstFreezingValues, val)
			lastFreezingValues = append(lastFreezingValues, int(math.NaN()))
		} else {
			firstFreezingValues = append(firstFreezingValues, int(math.NaN()))
			lastFreezingValues = append(lastFreezingValues, int(math.NaN()))
		}
		date = date.Add(time.Duration(24) * time.Hour)
	}
	return [][]int{firstFreezingValues, lastFreezingValues}
}

func getComparisonYear(c common.DawnCtx, request models.CSVRequest, dates []time.Time) []float64 {
	gddRequest := models.GddRequest{
		Year:       request.ComparisonYear,
		Product:    request.Product,
		Latitude:   request.Latitude,
		Longitude:  request.Longitude,
		Accumulate: true,
	}

	gddValues := GetGddValues(c, gddRequest).GddValues
	for len(gddValues) < 365 {
		gddValues = append(gddValues, math.NaN())
	}
	return gddValues
}

func getConfidenceInterval(request models.CSVRequest, dates []time.Time) [][]float64 {

	ciRequest := models.ConfidenceIntervalRequest{
		Interval:  request.Range,
		Product:   enums.GetProductFromString(request.Product),
		Latitude:  request.Latitude,
		Longitude: request.Longitude,
	}

	response := GetConfidenceInterval(ciRequest)

	return [][]float64{response.LowerBound, response.UpperBound}
}

func getCfsData(request models.CSVRequest, dates []time.Time) [][]float64 {

	gddRequest := models.GddRequest{
		Year:       request.Year,
		Product:    request.Product,
		Latitude:   request.Latitude,
		Longitude:  request.Longitude,
		Accumulate: true,
	}

	response := GetCfsGddValues(gddRequest)
	fullGddValues := []float64{}
	fullLowerBound := []float64{}
	fullUpperBound := []float64{}
	in := false
	after := false
	c := 0
	date := time.Date(response.FirstDate.Year(), time.January, 1, 0, 0, 0, 0, time.UTC)
	for i := 0; i < 366; i++ {
		if response.FirstDate.After(date) {
			fullGddValues = append(fullGddValues, math.NaN())
			fullLowerBound = append(fullLowerBound, math.NaN())
			fullUpperBound = append(fullUpperBound, math.NaN())
		} else {
			if in || response.FirstDate == date {
				if c >= len(response.GddValues) {
					in = false
					after = true
				} else {
					in = true
					fullGddValues = append(fullGddValues, response.GddValues[c])
					fullLowerBound = append(fullLowerBound, response.LowerBound[c])
					fullUpperBound = append(fullUpperBound, response.UpperBound[c])
					c += 1
				}
			} else if after {
				fullGddValues = append(fullGddValues, math.NaN())
				fullLowerBound = append(fullLowerBound, math.NaN())
				fullUpperBound = append(fullUpperBound, math.NaN())
			}
		}
		date = date.Add(time.Duration(24) * time.Hour)

	}
	return [][]float64{fullGddValues, fullLowerBound, fullUpperBound}
}

func getGefsData(request models.CSVRequest, dates []time.Time) []float64 {

	gddRequest := models.GddRequest{
		Year:       request.Year,
		Product:    request.Product,
		Latitude:   request.Latitude,
		Longitude:  request.Longitude,
		Accumulate: true,
	}

	response := GetGefsGddValues(gddRequest)
	fullGddValues := []float64{}
	in := false
	after := false
	c := 0
	date := time.Date(response.FirstDate.Year(), time.January, 1, 0, 0, 0, 0, time.UTC)
	for i := 0; i < 366; i++ {
		if response.FirstDate.After(date) {
			fullGddValues = append(fullGddValues, math.NaN())
		} else {
			if in || response.FirstDate == date {
				if c >= len(response.GddValues) {
					in = false
					after = true
				} else {
					in = true
					fullGddValues = append(fullGddValues, response.GddValues[c])
					c += 1
				}
			} else if after {
				fullGddValues = append(fullGddValues, math.NaN())
			}
		}
		date = date.Add(time.Duration(24) * time.Hour)

	}
	return fullGddValues
}

func pullData(c common.DawnCtx, request models.CSVRequest) CSVData {
	returnData := CSVData{}
	dates := fillDates()
	returnData.Date = dates

	if request.Analog {
		returnData.Analog = getAnalogYear(c.FiberCtx, request, dates)
	}
	if request.Cfs || request.CfsLower || request.CfsUpper {
		t := getCfsData(request, dates)
		returnData.Cfs = t[0]
		returnData.CfsLower = t[1]
		returnData.CfsUpper = t[2]
	}
	if request.Comparison {
		returnData.Comparison = getComparisonYear(c, request, dates)
	}
	if request.FirstFreezing || request.LastFreezing {
		t := getFreezingDates(request, dates)
		returnData.FirstFreezing = t[0]
		returnData.LastFreezing = t[1]
	}
	if request.Gefs {
		returnData.GEFS = getGefsData(request, dates)
	}
	if request.Maximum || request.Minimum {
		ci := getConfidenceInterval(request, dates)
		returnData.Maximum = ci[1]
		returnData.Minimum = ci[0]
	}
	if request.Normals {
		returnData.Normals = getNormals(request, dates)
	}
	if request.Primary {
		returnData.Primary = getPrimary(c, request, dates)
	}

	return returnData
}

func createRecords(keys []string, data CSVData) [][]string {
	records := [][]string{keys}
	for i := 1; i < 366; i++ {
		temp := []string{timeToString(data.Date[i-1])}
		if len(data.Analog) != 0 {
			temp = append(temp, fmt.Sprintf("%f", data.Analog[i-1]))
		}
		if len(data.Cfs) != 0 {
			temp = append(temp, fmt.Sprintf("%f", data.Cfs[i-1]))
			temp = append(temp, fmt.Sprintf("%f", data.CfsLower[i-1]))
			temp = append(temp, fmt.Sprintf("%f", data.CfsUpper[i-1]))
		}
		if len(data.Comparison) != 0 {
			temp = append(temp, fmt.Sprintf("%f", data.Comparison[i-1]))
		}
		if len(data.FirstFreezing) != 0 {
			temp = append(temp, fmt.Sprintf("%d", data.FirstFreezing[i-1]))
			temp = append(temp, fmt.Sprintf("%d", data.LastFreezing[i-1]))
		}
		if len(data.GEFS) != 0 {
			temp = append(temp, fmt.Sprintf("%f", data.GEFS[i-1]))
		}
		if len(data.Maximum) != 0 {
			temp = append(temp, fmt.Sprintf("%f", data.Maximum[i-1]))
			temp = append(temp, fmt.Sprintf("%f", data.Minimum[i-1]))
		}
		if len(data.Normals) != 0 {
			temp = append(temp, fmt.Sprintf("%f", data.Normals[i-1]))
		}
		if len(data.Primary) != 0 {
			temp = append(temp, fmt.Sprintf("%f", data.Primary[i-1]))
		}

		records = append(records, temp)
	}
	return records
}

func GetDataDownload(c common.DawnCtx, request models.CSVRequest) string {

	fileId := uuid.New()

	f, err := os.Create(fileId.String() + ".csv")

	if err != nil {
		panic(config.FILE_CREATION_ERROR.AddLogDetails("Could not create file"))
	}

	w := csv.NewWriter(f)

	keys := fillKeys(request)

	data := pullData(c, request)

	records := createRecords(keys, data)

	for _, record := range records {
		if err := w.Write(record); err != nil {
			log.Fatalln("error writing record to file", err)
		}
	}

	w.Flush()

	fileText, err := ioutil.ReadFile(fileId.String() + ".csv")
	f.Close()
	e := os.Remove(fileId.String() + ".csv")
	if e != nil {
		log.Fatal(e)
	}
	return string(fileText)
}
