package models

import (
	"fmt"
	"strconv"
	"time"

	"gitlab.cs.umd.edu/dawn/go-backend/dawn-gdd/config"
	"gitlab.cs.umd.edu/dawn/go-backend/dawn-gdd/models/enums"
	"gitlab.cs.umd.edu/dawn/go-backend/dawn-gdd/persistence/entities"
	"gitlab.cs.umd.edu/dawn/go-backend/dawn-gdd/utils"

	validation "github.com/go-ozzo/ozzo-validation"
	"github.com/gofiber/fiber/v2"
	"github.com/tgs266/dawn-go-common/errors"
)

type ConfidenceIntervalRequest struct {
	Product      enums.Product `json:"product"`
	Latitude     float64       `json:"latitude"`
	Longitude    float64       `json:"longitude"`
	Interval     float64       `json:"interval"`
	PlantingDate time.Time     `json:"plantingDate"`
	RepeatYears  int           `json:"repeatYears"`
}

type ConfidenceIntervalResposne struct {
	LowerBound       []float64 `json:"lower_bound,omitempty"`
	UpperBound       []float64 `json:"upper_bound,omitempty"`
	ClosestLatitude  float64   `json:"closest_latitude,omitempty"`
	ClosestLongitude float64   `json:"closest_longitude,omitempty"`
}

var IntervalConvert = map[float64]float64{
	80:   1.282,
	85:   1.440,
	90:   1.645,
	95:   1.960,
	99:   2.576,
	99.5: 2.807,
	99.9: 3.291,
}

var IntervalConvertPercentiles = map[float64][]float64{
	90:   {-1.645, -1.645},
	95:   {-1.960, 1.960},
	99:   {-2.576, 2.576},
	99.5: {-2.807, 2.807},
	99.9: {-3.291, 3.291},
}

func validateInterval(value interface{}) error {
	for k := range IntervalConvert {
		if k == value {
			return nil
		}
	}
	return fmt.Errorf("Interval not in allowed intervals")
}

func (r ConfidenceIntervalRequest) Validate() error {

	return validation.ValidateStruct(&r,
		validation.Field(&r.Latitude, validation.Required, validation.Min(-90.0), validation.Max(90.0)),
		validation.Field(&r.Longitude, validation.Required, validation.Min(-180.0), validation.Max(180.0)),
		validation.Field(&r.Interval, validation.Required, validation.By(validateInterval)),
	)
}

func (r ConfidenceIntervalRequest) Build(c *fiber.Ctx) ConfidenceIntervalRequest {
	repeatYears, errYears := strconv.Atoi(c.Query("repeatYears", "1"))

	if errYears != nil {
		panic(errors.NewBadRequest(nil).PutDetail("reason", "repeat years must be an integer"))
	}

	if repeatYears < 1 || repeatYears > 3 {
		panic(errors.NewBadRequest(nil).PutDetail("reason", "repeat years must be between 1 and 3 inclusive"))
	}
	product := c.Query("product", "NONE")
	latitude, _ := strconv.ParseFloat(c.Query("latitude", "-100000.0"), 64)
	longitude, _ := strconv.ParseFloat(c.Query("longitude", "-100000.0"), 64)
	interval, _ := strconv.ParseFloat(c.Query("interval", "-100000.0"), 64)
	pd := c.Query("plantingDate", utils.GetFirstOfTheYear().Format(time.RFC3339))
	plantingDate, errDate := time.Parse(time.RFC3339, pd)

	if errDate != nil {
		panic(errors.NewBadRequest(nil).PutDetail("reason", "no planting date provided"))
	}

	if errDate != nil && pd != "" {
		panic(errors.NewBadRequest(nil).PutDetail("reason", "date must be ISO8601 or RFC3339 format"))
	}

	rNew := ConfidenceIntervalRequest{
		Product:      enums.GetProductFromString(product),
		Latitude:     latitude,
		Longitude:    longitude,
		Interval:     interval,
		PlantingDate: plantingDate,
		RepeatYears:  repeatYears,
	}

	if rNew.Validate() != nil {
		panic(config.BAD_REQUEST)
	}

	return rNew
}

func (r ConfidenceIntervalRequest) BuildLocation() entities.Location {
	l := entities.Location{
		Type:        "Point",
		Coordinates: []float64{r.Longitude, r.Latitude},
	}
	return l
}
