package persistence

import (
	"context"

	"github.com/tgs266/dawn-go-common/common"
	"gitlab.cs.umd.edu/dawn/go-backend/dawn-gdd/config"
	"gitlab.cs.umd.edu/dawn/go-backend/dawn-gdd/persistence/entities"
	"gitlab.cs.umd.edu/dawn/go-backend/dawn-gdd/persistence/repositories"

	"go.mongodb.org/mongo-driver/bson"
	"go.mongodb.org/mongo-driver/bson/primitive"
	"go.mongodb.org/mongo-driver/mongo/options"
)

var Session *common.DBSession

func GddRepository() repositories.GddRepository {
	return repositories.NewGddRepository(Session)
}

func NomadsRepository() repositories.NomadsRepository {
	return repositories.NewNomadsRepository(Session)
}

// so we can replace during testing
var FreezingDatesRepository = func() repositories.FreezingDatesRepository {
	return repositories.NewFreezingDatesRepository(Session)
}

func buildLocationRequest(location entities.Location, year *int) bson.M {
	var filter bson.M
	if year != nil {
		filter = bson.M{
			"location": bson.M{
				"$nearSphere": bson.M{
					"$geometry":    location,
					"$maxDistance": 50000,
				},
			},
			"year": *year,
		}
	} else {
		filter = bson.M{
			"location": bson.M{
				"$nearSphere": bson.M{
					"$geometry":    location,
					"$maxDistance": 50000,
				},
			},
		}
	}
	return filter
}

func buildLocationRequestLarger(location entities.Location, year *int) bson.M {
	var filter bson.M
	if year != nil {
		filter = bson.M{
			"location": bson.M{
				"$nearSphere": bson.M{
					"$geometry":    location,
					"$maxDistance": 100000,
				},
			},
			"year": *year,
		}
	} else {
		filter = bson.M{
			"location": bson.M{
				"$nearSphere": bson.M{
					"$geometry":    location,
					"$maxDistance": 100000,
				},
			},
		}
	}
	return filter
}

func FindCultivarByName(name string) entities.Cultivar {
	coll := Session.Collection("dssat")

	filter := bson.M{
		"var_name": bson.M{
			"$regex": primitive.Regex{Pattern: name, Options: "i"},
		},
	}

	var results entities.Cultivar

	err := coll.FindOne(Session.Ctx, filter).Decode(&results)
	if err != nil {
		panic(config.NO_DATA_FOUND)
	}

	return results
}

func FindCultivarsByName() []entities.Cultivar {
	coll := Session.Collection("dssat")

	var results []entities.Cultivar

	options := options.Find()

	cursor, err := coll.Find(Session.Ctx, bson.D{}, options)
	if err != nil {
		panic(config.NO_DATA_FOUND)
	}

	for cursor.Next(context.TODO()) {

		var elem entities.Cultivar
		err := cursor.Decode(&elem)
		if err != nil {
			panic(config.NO_DATA_FOUND)
		}

		results = append(results, elem)
	}

	return results
}
