package repositories

import (
	"context"
	errs "errors"

	"github.com/tgs266/dawn-go-common/common"
	"github.com/tgs266/dawn-go-common/errors"
	"gitlab.cs.umd.edu/dawn/go-backend/dawn-gdd/models"
	"gitlab.cs.umd.edu/dawn/go-backend/dawn-gdd/persistence/entities"
	"go.mongodb.org/mongo-driver/bson"
	"go.mongodb.org/mongo-driver/mongo"
	"go.mongodb.org/mongo-driver/mongo/options"
)

type GddRepository interface {
	// FindCurrentGddByLocation finds the current GDD value for a given location.
	FindCurrentGddByLocation(ctx context.Context, location entities.Location) entities.Gdd

	// FindGddByLocationAndYear finds the GDD value for a given location and year.
	FindGddByLocationAndYear(ctx context.Context, year int, location entities.Location) entities.Gdd

	// FindGddsOverNormalsRangeByLocation finds the GDD values for a given location over a range of years.
	FindGddsOverNormalsRangeByLocation(ctx context.Context, location entities.Location) []entities.Gdd

	// FindAnalogYearByLocation finds the analog year for a given location.
	FindAnalogYearByLocation(ctx context.Context, location entities.Location) models.AnalogResponse
}

type gddRepositoryImpl struct {
	session *common.DBSession
}

// NewGddRepository creates a new GddRepository instance.
func NewGddRepository(session *common.DBSession) *gddRepositoryImpl {
	return &gddRepositoryImpl{
		session: session,
	}
}

// buildLocationRequest builds a BSON filter for a given location and year.
// If the year is nil, only the location filter will be included.
func buildLocationRequest(location entities.Location, year *int) bson.M {
	filter := bson.M{
		"location": bson.M{
			"$nearSphere": bson.M{
				"$geometry":    location,
				"$maxDistance": 50000,
			},
		},
	}

	if year != nil {
		filter["year"] = *year
	}

	return filter
}

func buildLocationRequestLarger(location entities.Location, year *int) bson.M {
	filter := bson.M{
		"location": bson.M{
			"$nearSphere": bson.M{
				"$geometry":    location,
				"$maxDistance": 100000,
			},
		},
	}

	if year != nil {
		filter["year"] = *year
	}

	return filter
}

// FindCurrentGddByLocation finds the current Gdd based on the given location.
func (g *gddRepositoryImpl) FindCurrentGddByLocation(ctx context.Context, location entities.Location) entities.Gdd {
	coll := g.session.Collection("gdd_current")

	filter := buildLocationRequest(location, nil)

	var gdd entities.Gdd
	if err := coll.FindOne(ctx, filter).Decode(&gdd); err != nil {
		if errs.Is(err, mongo.ErrNoDocuments) {
			panic(errors.NewNotFound(err).PutDetail("reason", "gdd_current"))
		}
		panic(errors.NewInternal(err))
	}
	return gdd
}

// FindGddByLocationAndYear finds the Gdd based on the given location and year.
func (g *gddRepositoryImpl) FindGddByLocationAndYear(ctx context.Context, year int, location entities.Location) entities.Gdd {
	coll := g.session.Collection("gdd")

	filter := buildLocationRequest(location, &year)

	var gdds entities.Gdd
	if err := coll.FindOne(ctx, filter).Decode(&gdds); err != nil {
		if errs.Is(err, mongo.ErrNoDocuments) {
			panic(errors.NewNotFound(err).PutDetail("reason", "gdd"))
		}
		panic(errors.NewInternal(err))
	}
	return gdds
}

// FindGddsOverNormalsRangeByLocation returns a slice of Gdd entities for the given location
// within the year range from 1991 to 2020. It panics if an error occurs or if no Gdd entities are found.
func (g *gddRepositoryImpl) FindGddsOverNormalsRangeByLocation(ctx context.Context, location entities.Location) []entities.Gdd {
	coll := g.session.Collection("gdd")

	filter := buildLocationRequest(location, nil)
	filter["year"] = bson.M{"$gte": 1991, "$lte": 2020}

	cursor, err := coll.Find(ctx, filter, options.Find().SetLimit(30))
	if err != nil {
		panic(errors.NewInternal(err).PutDetail("reason", "gdd"))
	}

	var gdds []entities.Gdd
	if err = cursor.All(ctx, &gdds); err != nil {
		panic(errors.NewInternal(err))
	}

	if len(gdds) == 0 {
		panic(errors.NewNotFound(nil).PutDetail("reason", "gdd"))
	}
	return gdds
}

func (g *gddRepositoryImpl) FindAnalogYearByLocation(ctx context.Context, location entities.Location) models.AnalogResponse {
	gdds := g.FindCurrentGddByLocation(ctx, location)
	return models.AnalogResponse{
		ClosestLatitude:  gdds.Location.Coordinates[1],
		ClosestLongitude: gdds.Location.Coordinates[0],
		AnalogYear:       gdds.AnalogYear,
	}
}
