package repositories

import (
	"context"
	errs "errors"
	"time"

	"github.com/bradfitz/slice"
	"github.com/tgs266/dawn-go-common/common"
	"github.com/tgs266/dawn-go-common/errors"
	"gitlab.cs.umd.edu/dawn/go-backend/dawn-gdd/persistence/entities"
	"go.mongodb.org/mongo-driver/bson"
	"go.mongodb.org/mongo-driver/mongo"
	"go.mongodb.org/mongo-driver/mongo/options"
)

type NomadsRepository interface {
	FindCfsByLocation(ctx context.Context, location entities.Location) []entities.CfsGdd
	FindCfsByLocationWithExpand(ctx context.Context, location entities.Location, areaExpand int) []entities.CfsGdd
	FindCfsByLocationWithExpandByPlantingDate(ctx context.Context, plantingDate time.Time, location entities.Location, areaExpand int) []entities.CfsGdd
	FindCfsAverageByLocation(ctx context.Context, location entities.Location) entities.CfsGdd
	FindGefsByLocation(ctx context.Context, location entities.Location) []entities.GefsGdd
}

type nomadsRepositoryImpl struct {
	session *common.DBSession
}

func NewNomadsRepository(session *common.DBSession) NomadsRepository {
	return &nomadsRepositoryImpl{
		session: session,
	}
}

func (g *nomadsRepositoryImpl) FindGefsByLocation(ctx context.Context, location entities.Location) []entities.GefsGdd {
	coll := g.session.Collection("gefs")

	filter := buildLocationRequestLarger(location, nil)

	var results []entities.GefsGdd

	options := options.Find()

	count := 10
	options.SetLimit(int64(count))

	cursor, err := coll.Find(ctx, filter, options)
	if err != nil {
		panic(errors.NewInternal(err).PutDetail("reason", "gefs"))
	}

	if err = cursor.All(ctx, &results); err != nil {
		panic(errors.NewInternal(err))
	}

	if len(results) == 0 {
		panic(errors.NewNotFound(nil).PutDetail("reason", "gefs"))
	}

	slice.Sort(results[:], func(i, j int) bool {
		return results[i].Date.Time().Unix() < results[j].Date.Time().Unix()
	})
	return results
}

// returns -1 member from cfs, which is the average of all other members
func (g *nomadsRepositoryImpl) FindCfsAverageByLocation(ctx context.Context, location entities.Location) entities.CfsGdd {
	coll := g.session.Collection("cfs")

	filter := buildLocationRequestLarger(location, nil)
	// -1 is the mean of all
	filter["member"] = -1
	var cfs entities.CfsGdd
	err := coll.FindOne(ctx, filter).Decode(&cfs)

	if errs.Is(err, mongo.ErrNoDocuments) {
		panic(errors.NewNotFound(err).PutDetail("reason", "cfs"))
	} else if err != nil {
		panic(errors.NewInternal(err))
	}

	return cfs
}

// this gets the cfs for a location, and using the parameter areaExpand allows you to expand the area of the data collection to get more samples
func (g *nomadsRepositoryImpl) findCfsByLocationInternal(ctx context.Context, location entities.Location, areaExpand int) []entities.CfsGdd {
	count := g.FindCfsAverageByLocation(ctx, location).Count

	coll := g.session.Collection("cfs")

	filter := buildLocationRequestLarger(location, nil)
	cursor, err := coll.Find(ctx, filter, options.Find().SetLimit(int64(count*areaExpand)))

	var gs []entities.CfsGdd

	if err != nil {
		panic(errors.NewInternal(err).PutDetail("reason", "cfs"))
	}

	if err = cursor.All(ctx, &gs); err != nil {
		panic(errors.NewInternal(err))
	}

	if len(gs) == 0 {
		panic(errors.NewNotFound(nil).PutDetail("reason", "cfs"))
	}

	return gs
}

func (g *nomadsRepositoryImpl) FindCfsByLocation(ctx context.Context, location entities.Location) []entities.CfsGdd {
	return g.findCfsByLocationInternal(ctx, location, 1)
}

func (g *nomadsRepositoryImpl) FindCfsByLocationWithExpand(ctx context.Context, location entities.Location, areaExpand int) []entities.CfsGdd {
	return g.findCfsByLocationInternal(ctx, location, areaExpand)
}

func (g *nomadsRepositoryImpl) FindCfsByLocationWithExpandByPlantingDate(ctx context.Context, plantingDate time.Time, location entities.Location, areaExpand int) []entities.CfsGdd {
	count := g.FindCfsAverageByLocation(ctx, location).Count

	coll := g.session.Collection("cfs")

	filter := buildLocationRequestLarger(location, nil)
	filter["date"] = bson.M{"$gte": plantingDate}
	cursor, err := coll.Find(ctx, filter, options.Find().SetLimit(int64(count*areaExpand)))

	var gs []entities.CfsGdd

	if err != nil {
		panic(errors.NewInternal(err).PutDetail("reason", "cfs"))
	}

	if err = cursor.All(ctx, &gs); err != nil {
		panic(errors.NewInternal(err))
	}

	if len(gs) == 0 {
		panic(errors.NewNotFound(nil).PutDetail("reason", "cfs"))
	}

	return gs
}
