package services

import (
	"context"
	"time"

	"gitlab.cs.umd.edu/dawn/go-backend/dawn-gdd/models"
	"gitlab.cs.umd.edu/dawn/go-backend/dawn-gdd/models/enums"
	"gitlab.cs.umd.edu/dawn/go-backend/dawn-gdd/persistence"
	"gitlab.cs.umd.edu/dawn/go-backend/dawn-gdd/persistence/entities"
	"gitlab.cs.umd.edu/dawn/go-backend/dawn-gdd/utils"
	"gonum.org/v1/gonum/stat"
)

func GetGddValues(ctx context.Context, request models.GddRequest) models.GddResponse {
	product := enums.GetProductFromString(request.Product)
	var gdds entities.Gdd
	if request.PlantingDate.After(time.Now()) {
		return models.GddResponse{
			Product:          product.Name,
			ClosestLatitude:  request.Latitude,
			ClosestLongitude: request.Longitude,
			GddValues:        []float64{},
		}
	} else if request.PlantingDate.Before(time.Now()) && request.PlantingDate.Year() != time.Now().Year() {
		gdds = persistence.GddRepository().FindGddByLocationAndYear(ctx, request.PlantingDate.Year(), request.BuildLocation())
	} else {
		gdds = persistence.GddRepository().FindCurrentGddByLocation(ctx, request.BuildLocation())
	}

	// if request.PlantingDate.Year() >= time.Now().Year() {
	pdInt := request.PlantingDate.YearDay() - 1
	// fmt.Println(pdInt)
	if pdInt < len(gdds.MaxTemps) {
		gdds.MinTemps = gdds.MinTemps[pdInt:]
		gdds.MaxTemps = gdds.MaxTemps[pdInt:]
	}

	returnGdds := models.GddResponse{
		Product:          product.Name,
		ClosestLatitude:  gdds.Location.Coordinates[1],
		ClosestLongitude: gdds.Location.Coordinates[0],
		GddValues:        utils.CalculateGddValues(gdds.MinTemps, gdds.MaxTemps, product, request.Accumulate),
		LastDate:         gdds.LastDate.Time(),
	}
	return returnGdds
}

func GetNormalValues(ctx context.Context, request models.GddRequest, repeatYears int) models.GddResponse {
	product := enums.GetProductFromString(request.Product)
	gs := persistence.GddRepository().FindGddsOverNormalsRangeByLocation(ctx, request.BuildLocation())
	var returnGdds models.GddResponse
	var gdds []float64
	rows := [][]float64{}

	// get the int of the planting date. If the date is less than the first date, we do nothing
	// otherwise, we adjust
	// need to do before because of accumulations
	sliceDateInt := request.PlantingDate.YearDay() - 1
	if sliceDateInt < 0 {
		sliceDateInt = 0
	}

	for i := sliceDateInt; i < len(gs[0].MinTemps)*repeatYears; i++ { //; i++ {
		row := make([]float64, len(gs))
		idx := i
		if i >= len(gs[0].MinTemps) {
			idx -= len(gs[0].MinTemps)
		}
		for j := 0; j < len(gs); j++ {
			row[j] = utils.CalculateSingleGdd(gs[j].MinTemps[idx], gs[j].MaxTemps[idx], product)
			if request.Accumulate && i > sliceDateInt {
				row[j] += rows[len(rows)-1][j]
			}
		}
		rows = append(rows, row)
		mean := stat.Mean(row, nil)
		gdds = append(gdds, mean)
	}

	returnGdds = models.GddResponse{
		Product:          product.Name,
		ClosestLatitude:  gs[0].Location.Coordinates[1],
		ClosestLongitude: gs[0].Location.Coordinates[0],
		GddValues:        gdds,
	}

	return returnGdds
}
