package services

import (
	"context"
	"encoding/json"
	"fmt"
	"io"
	"io/ioutil"
	"math"
	"math/rand"
	"os"
	"os/exec"
	"strconv"
	"strings"
	"time"

	"gitlab.cs.umd.edu/dawn/go-backend/dawn-gdd/config"
	"gitlab.cs.umd.edu/dawn/go-backend/dawn-gdd/models"
	"gitlab.cs.umd.edu/dawn/go-backend/dawn-gdd/models/enums"
	"gitlab.cs.umd.edu/dawn/go-backend/dawn-gdd/persistence"
	"gitlab.cs.umd.edu/dawn/go-backend/dawn-gdd/persistence/entities"
	"gitlab.cs.umd.edu/dawn/go-backend/dawn-gdd/utils"
)

type CornMaturityConfig struct {
	PlantDate int       `json:"plant_date"`
	Tmins     []float64 `json:"tmins"`
	Tmaxs     []float64 `json:"tmaxs"`
	Srads     []float64 `json:"srads"`
	Units     string    `json:"units"`
}

func GetCultivars(ctx context.Context, request models.CultivarRequest) models.CultivarResponse {
	if request.Name != "" {
		elem := persistence.FindCultivarByName(request.Name)
		cultivar := models.CreateCultivar(elem)
		return models.CultivarResponse{
			Cultivars: []models.Cultivar{cultivar},
		}
	} else {
		elems := persistence.FindCultivarsByName()
		var cultivars []models.Cultivar
		for _, e := range elems {
			cultivars = append(cultivars, models.CreateCultivar(e))
		}
		return models.CultivarResponse{
			Cultivars: cultivars,
		}
	}
}

func getFullYearData(ctx context.Context, latitude float64, longitude float64, plantDate time.Time) CornMaturityConfig {
	location := entities.Location{
		Type:        "Point",
		Coordinates: []float64{longitude, latitude},
	}

	observed := persistence.GddRepository().FindCurrentGddByLocation(ctx, location)
	predicted := persistence.NomadsRepository().FindCfsAverageByLocation(ctx, location)

	tmins := append(observed.MaxTemps, predicted.MaxTemps...)
	tmaxs := append(observed.MinTemps, predicted.MinTemps...)
	srads := append(observed.SRAD, predicted.Srad...)

	pd := plantDate.YearDay() - 1

	if plantDate.Year()%4 == 0 && plantDate.Year()%100 != 0 || plantDate.Year()%400 == 0 {
		pd -= 1
	}

	return CornMaturityConfig{
		PlantDate: pd,
		Tmins:     tmins,
		Tmaxs:     tmaxs,
		Srads:     srads,
		Units:     "F",
	}
}

func CalculateMaturity(ctx context.Context, request models.CornMaturityRequest) models.CornMaturityResponse {

	cfg := getFullYearData(ctx, request.Latitude, request.Longitude, request.PlantDate)
	fileId := rand.Intn(99-10) + 10

	file, _ := json.MarshalIndent(cfg, "", " ")

	_ = ioutil.WriteFile("lib/maize/"+strconv.Itoa(fileId)+".json", file, 0644)

	original, err := os.Open("lib/maize/DSSAT.INP")
	if err != nil {
		panic(err)
	}
	new, err := os.Create("lib/maize/DSSAT" + strconv.Itoa(fileId) + ".INP")
	if err != nil {
		panic(err)
	}
	_, err = io.Copy(new, original)
	if err != nil {
		panic(err)
	}
	new.Close()
	original.Close()

	f, err := os.OpenFile("lib/maize/DSSAT"+strconv.Itoa(fileId)+".INP", os.O_APPEND|os.O_WRONLY|os.O_CREATE, 0600)
	if err != nil {
		panic(err)
	}
	cultivarLineStart := fmt.Sprintf("%s %s",
		request.VarNum,
		request.VarName,
	)
	// ECONUM + ... MUST START AFTER 24th CHARACTER
	cultivarLineStart += strings.Repeat(" ", 24-len(cultivarLineStart))
	cultivarLineEnd := fmt.Sprintf("%s %s %s %s %s %s %s",
		request.EcoNum,
		fmt.Sprintf("%f", request.P1)[:5],
		fmt.Sprintf("%f", request.P2)[:5],
		fmt.Sprintf("%f", request.P5)[:5],
		fmt.Sprintf("%f", request.G2)[:5],
		fmt.Sprintf("%f", request.G3)[:5],
		fmt.Sprintf("%f", request.PHINT)[:5],
	)

	cultivarLine := cultivarLineStart + cultivarLineEnd
	if _, err = f.WriteString(cultivarLine); err != nil {
		panic(err)
	}
	f.Close()

	cmd := exec.Command("bash", "-c", "python3 execute.py --input "+strconv.Itoa(fileId)+".json --config "+strconv.Itoa(fileId)+" --predicted")
	cmd.Dir = "./lib/maize"
	out, _ := cmd.Output()

	value := strings.TrimSpace(string(out))
	if value == "-1" || strings.HasPrefix(value, "A") {
		panic(config.BAD_REQUEST.AddLogDetails("Script returned: " + value))
	}

	year, _ := strconv.Atoi(value[:4])
	doy, _ := strconv.Atoi(value[4:])

	date := time.Date(year, time.January, 1, 0, 0, 0, 0, time.UTC)
	if year%4 == 0 && year%100 != 0 || year%400 == 0 {
		doy -= 1
	}
	date = date.AddDate(0, 0, doy)

	os.Remove("lib/maize/" + strconv.Itoa(fileId) + ".json")
	os.Remove("lib/maize/DSSAT" + strconv.Itoa(fileId) + ".INP")

	return models.CornMaturityResponse{
		HarvestDate: date,
	}
}

func CalculateRMMaturity(ctx context.Context, request models.CornRMMaturityRequest) models.CornMaturityResponse {
	fyData := getFullYearData(ctx, request.Latitude, request.Longitude, request.PlantDate)
	gdds := utils.CalculateGddValues(fyData.Tmins, fyData.Tmaxs, enums.ProductType.CORN, false)

	start := request.PlantDate.YearDay() - 1
	year := request.PlantDate.Year()
	if year%4 == 0 && year%100 != 0 || year%400 == 0 {
		start -= 1
	}

	acc := 0.0
	lastAcc := 0.0
	expected := float64(((request.RelativeMaturity - 95.0) * 22.0) + 2375.0)
	found := 0

	for i := start; i < len(gdds); i++ {
		lastAcc = acc
		acc += gdds[i]
		found = i
		if math.Abs(acc-expected) > math.Abs(lastAcc-expected) {
			break
		}
	}

	if year%4 == 0 && year%100 != 0 || year%400 == 0 {
		found += 1
	}

	date := request.PlantDate.AddDate(0, 0, found-start)

	return models.CornMaturityResponse{
		HarvestDate: date,
	}

}
