package dispatch

import (
	"fmt"
)

/*
Helper type for dispatching operations in a clean manner

Please add a defer to any function that uses these

Example:

	gddThread := dispatch.New[models.GddResponse]()
	go func() {
		defer func() { gddThread.Recover(recover()) }()
		gdds := GetGddValues(ctx, gddReq)
		gddThread.Result(gdds)
	}()

	gdds, err := gddThread.Get()
	if err != nil {
		panic(err)
	}
*/
type Thread[T any] struct {
	result chan ThreadData[T]
}

type ThreadData[T any] struct {
	err    error
	result T
}

func New[T any]() *Thread[T] {
	return &Thread[T]{
		result: make(chan ThreadData[T]),
	}
}

func (t *Thread[T]) Error(err error) {
	t.result <- ThreadData[T]{
		err: err,
	}
}

func (t *Thread[T]) Result(r T) {
	result := ThreadData[T]{
		result: r,
		err:    nil,
	}

	t.result <- result
}

func (t *Thread[T]) Recover(r any) {
	if e, ok := r.(error); ok {
		t.Error(e)
	} else {
		t.Error(fmt.Errorf("unknown error"))
	}
}

// will block
func (t *Thread[T]) Get() (T, error) {
	select {
	case result := <-t.result:
		return result.result, result.err
	}
}
