package utils

import (
	"gitlab.cs.umd.edu/dawn/go-backend/dawn-gdd/models/enums"
)

// CalculateSingleGdd calculates the Growing Degree Days (GDD) value for a given temperature and product.
func CalculateSingleGdd(minTemp, maxTemp float64, product enums.Product) float64 {
	// If the product is corn, clip the minimum and maximum temperature values to a range of 50-86.
	if product.Name == "CORN" {
		minTemp = ClipFloat(minTemp, 50, 86)
		maxTemp = ClipFloat(maxTemp, 50, 86)
	}

	// Calculate the GDD
	mean := (maxTemp + minTemp) / 2.0
	value := mean - product.BaseTemp

	// If the product is corn, return the value without further processing
	if product.Name == "CORN" {
		return value
	}

	// Clip the value to a minimum of 0
	value = ClipMinFloat(value, 0)
	return value
}

func CalculateGddValues(minTemps []float64, maxTemps []float64, product enums.Product, accumulate bool) []float64 {
	var returnList []float64
	for i := 0; i < len(minTemps); i++ {
		value := CalculateSingleGdd(minTemps[i], maxTemps[i], product)
		if accumulate && i > 0 {
			value += returnList[len(returnList)-1]
		}
		returnList = append(returnList, value)
	}
	return returnList
}

// anomaly adjusted, https://library.wmo.int/doc_num.php?explnum_id=10314 page 22
func CalculateGddValuesCfsNormed(minTemps []float64, maxTemps []float64, product enums.Product, cfsMinTemps []float64, cfsMaxTemps []float64, normalMean []float64) []float64 {
	var returnList []float64
	for i := 0; i < len(minTemps); i++ {
		value := CalculateSingleGdd(minTemps[i], maxTemps[i], product)
		cfsValue := CalculateSingleGdd(cfsMinTemps[i], cfsMaxTemps[i], product)
		value = value + (normalMean[i] - cfsValue)
		value = ClipMinFloat(value, 0)
		returnList = append(returnList, value)
	}
	return returnList
}

// adjust all values in array by a constant value
func Adjust[T float64 | float32 | int | int64](array []T, value T) []T {
	for i := range array {
		array[i] += value
	}
	return array
}
