//
package main

/*
 Two main files are ../fuse.go and ../fs/serve.go
*/

import (
	"encoding/binary"
	"fmt"
	"github.com/golang/protobuf/proto"
	"p2/pb"

	"bazil.org/fuse"
	"os"
	"time"
)

type DNode struct {
	Name       string
	Attrs      fuse.Attr
	Version    int
	PrevSig    string
	ChildSigs  map[string]string
	DataBlocks []string
}

//=====================================================================

func (n *DNode) isDir() bool {
	return (n.Attrs.Mode & os.ModeDir) != 0
}

func importProtoNode(bytes []byte, out *DNode) error {
	in := new(pb.Node)
	if err := proto.Unmarshal(bytes, in); err != nil {
		return err
	}

	out.Name = in.Name
	out.Version = int(in.Version)
	out.PrevSig = in.PrevSig

	out.Attrs.Valid = time.Duration(in.Attrs.Valid)
	out.Attrs.Inode = in.Attrs.Inode
	out.Attrs.Size = in.Attrs.Size
	out.Attrs.Blocks = in.Attrs.Blocks
	out.Attrs.Mode = os.FileMode(in.Attrs.FileMode)
	out.Attrs.Nlink = in.Attrs.Nlink
	out.Attrs.Uid = in.Attrs.Uid
	out.Attrs.Gid = in.Attrs.Gid
	out.Attrs.Rdev = in.Attrs.Rdev
	out.Attrs.Flags = in.Attrs.Flags
	out.Attrs.BlockSize = in.Attrs.BlockSize

	out.Attrs.Mtime = time.Unix(0, int64(in.Attrs.Mtime))
	out.Attrs.Ctime = time.Unix(0, int64(in.Attrs.Ctime))
	out.Attrs.Crtime = time.Unix(0, int64(in.Attrs.Crtime))
	out.Attrs.Atime = time.Unix(0, int64(in.Attrs.Atime))

	out.ChildSigs = in.ChildSigs
	out.DataBlocks = in.DataBlocks

	return nil
}

//=====================================================================

func main() {
	if len(os.Args) < 2 {
		fmt.Println("USAGE: readlog <log path>\n")
	} else {
		bytes, err := os.ReadFile(os.Args[1])
		if err != nil {
			fmt.Printf("no metalog\n")
		} else {

			for i := 0; i < len(bytes); {
				len := int(binary.LittleEndian.Uint32(bytes[i : i+4]))

				n := new(DNode)
				nodebytes := bytes[i+4 : i+4+len]

				err = importProtoNode(nodebytes, n)

				i += 4 + len

				if n.isDir() {
					kids := []string{}
					for k, _ := range n.ChildSigs {
						kids = append(kids, k)
					}
					fmt.Printf("%q %v\n", n.Name, kids)
				} else {
					fmt.Printf("%q (%d bytes)\n", n.Name, n.Attrs.Size)
				}
			}
		}
	}
}
