//
package main

/*

 */

import (
	"fmt"
	. "github.com/mattn/go-getopt"
	"golang.org/x/net/context"
	"google.golang.org/grpc"
	"log"
	"os"
	"p3/pb"
	"strconv"
	"time"
)

//=============================================================================

const (
	CHUNK = iota
	MERKLE
	LOG
	DIE
)

var (
	debug = false
)

func main() {
	var c int
	localserver := "localhost:8000"
	remoteserver := "localhost:8001"
	height := 3
	cmd := MERKLE
	chunkSig := ""

	for {
		if c = Getopt("c:dDh:mlR:L:"); c == EOF {
			break
		}

		switch c {
		case CHUNK:
		case 'c':
			cmd = CHUNK
			chunkSig = OptArg
		case 'd':
			debug = !debug
		case 'D':
			cmd = DIE
		case 'h':
			height, _ = strconv.Atoi(OptArg)
		case 'l':
			cmd = LOG
		case 'm':
			cmd = MERKLE
		case 'L':
			localserver = OptArg
		case 'R':
			remoteserver = OptArg
		default:
			println("usage: main.go [-d | -h <tree height> | -R <remote:port> | -S <server:port> | -m | -l ]", c)
			os.Exit(1)
		}
	}

	initRPC(localserver)

	fmt.Printf("\nStartup up with cmd %v debug %v, local %s, remote %s, height: %d\n\n", cmd, debug, localserver, remoteserver, height)
	switch cmd {
	case DIE:
		requestDie(localserver)
	case CHUNK:
		data := requestChunk(chunkSig)
		if data == nil {
			pout("Chunk request %q failed\n", chunkSig)
		} else {
			pout("chunk %q returned %d bytes\n", chunkSig, len(data))
		}
	case MERKLE:
		requestSyncChunks(height, remoteserver)
	case LOG:
		requestSyncLog(localserver, remoteserver)
	}
}

var (
	server pb.ReplicaClient
)

func initRPC(address string) {
	// Set up a connection to the server.
	conn, err := grpc.Dial(address, grpc.WithInsecure())
	if err != nil {
		log.Fatalf("did not connect: %v", err)
	}
	server = pb.NewReplicaClient(conn)
}

//=====================================================================

func requestSyncChunks(depth int, rserver string) []byte {
	ctx, cancel := context.WithTimeout(context.Background(), 60*time.Second)
	defer cancel()

	pout("Syncing w/ remote server %q, depth %d\n", rserver, depth)
	reply, err := server.SyncBlocks(ctx, &pb.SyncBlocksRequest{Sig: "", Depth: depth, RemoteAddress: rserver})
	if err != nil {
		log.Fatalf("Sync error: %v\n", err)
	}
	log.Printf("Sync succeeds: %v\n", reply)
	return []byte{}
}

//=====================================================================
func requestDie(s string) {
	ctx, cancel := context.WithTimeout(context.Background(), 60*time.Second)
	defer cancel()

	pout("Telling %q to die\n", s, 3)
	_, err := server.Die(ctx, &pb.DieRequest{})
	if err != nil {
		log.Fatalf("Die error: %v\n", err)
	}
	log.Printf("Die succeeds\n")
}

//=====================================================================

func requestSyncLog(lserver, rserver string) {
	ctx, cancel := context.WithTimeout(context.Background(), 60*time.Second)
	defer cancel()

	pout("Grabbing remote log from %q to %q\n", rserver, lserver)
	reply, err := server.SyncLog(ctx, &pb.SyncLogRequest{RemoteAddress: rserver})
	if err != nil {
		log.Fatalf("Sync error: %v\n", err)
	}
	log.Printf("Read %d-byte log from %s\n", reply.Length, rserver)
}

//=====================================================================

func requestChunk(sig string) []byte {
	ctx, cancel := context.WithTimeout(context.Background(), time.Second)
	defer cancel()

	reply, err := server.ChunkGet(ctx, &pb.ChunkRequest{Sig: sig})
	if err != nil {
		log.Fatalf("ERROR chunkget: %v", err)
		return nil
	}
	pout("gRPC GOT %d bytes for %q\n", len(reply.Data), reply.Sig)
	return reply.Data
}

//=====================================================================

func pout(s string, args ...interface{}) {
	if !debug {
		return
	}
	log.Printf(s, args...)
}

func pif(show bool, s string, args ...interface{}) {
	if !show {
		return
	}
	log.Printf(s, args...)
}

func perr(s string, args ...interface{}) {
	log.Printf(s, args...)
}
