package Site::Cacti;

use strict;
use warnings;

use base 'Resmon::Module';

use Resmon::ExtComm qw(run_command cache_command);

=pod

=head1 Site::Cacti

Site::Cacti is a Cacti log parser to summarize failed datasets number

=head1 SYNOPSIS

 Site::Command {
     some_check_name: noop;
 }

=head1 DESCRIPTION

This module parses Cacti Log and gets total numbers of Datasets and 
how many of them failed during last cacti run.

=head1 CONFIGURATION

=over

=item check_name

=back

=cut


my $DEBUG=0;
my $SLEEP=5;

sub handler {
    my $self = shift;
    my $config = $self->{config}; # All configuration is in here
    my $check_name = $self->{check_name}; # The check name is in here
    my $SPAN=(5*60); # seconds
    my $LOG='/var/log/cacti/cacti.log';
    my $END=`tail -1 $LOG`; chomp $END;
    $END = $END =~ m<^(\d{4}/\d{1,2}/\d{1,2} \d\d:\d\d)> ? $1 :
        die "problem with tail $LOG: strange date/time in the line \n $END";
    my $M = substr $END, -1;
    $M = $M < 5 ? 0 : 5;
    my $START=$END;
    $START =~ s/.$/$M/;
    open (my $log, "<$LOG") or die "can't open <$LOG: $!";
=pod example of the log line:
2018/11/02 14:45:05 - SPINE: Poller[1] Device[14] TH[1] DS[1341] SNMP: v2: 172.26.69.124, dsname: traffic_out, oid: .1.3.6.1.2.1.31.1.1.1.10.669, value: 93577117
=cut
    while (<$log>) {
      last if($_ =~ m<^$START>);
    }
    my ($errors,$items,$hosts,$rrds,$datasets,$complete,$time,$error) = (0,0,0,0,0,0,0.0,'');
    until ($complete) {
        while (<$log>) {
=pod example of errors reported:
2018/11/02 17:00:05 - SPINE: Poller[1] WARNING: Invalid Response(s), Errors[1] Device[9] Thread[1] DS[903]
=cut
          if(/SPINE:\s+Poller\[\d+\]\s+WARNING:.*Errors\[(\d+)\]\s+Device\[(\d+)\].*\s+DS\[([ 0-9,]+)\]/) {
            print STDERR "errors: num=$1, Device\[$2], DSes=$3\n" if $DEBUG;
            $errors += $1;
          }
=pod example of DSes  reported:
2018/11/05 15:00:05 - SPINE: Poller[1] Device[32] TH[1] NOTE: There are '398' Polling Items for this Device
=cut
          elsif(/SPINE:\s+Poller\[\d+\]\s+Device\[(\d+)\]\s+TH\[\d+\]\s+NOTE: There are '(\d+)' Polling Items for this Device/) {
            $hosts += 1;
            $items += $2;
            print STDERR "There are $2 items for Device\[$1\]\n" if $DEBUG;
          }
=pod example of the line to be expected at the end:
2018/11/07 00:00:06 - SYSTEM STATS: Time:5.4350 Method:spine Processes:2 Threads:2 Hosts:56 HostsPerProcess:28 DataSources:9131 RRDsProcessed:3323
=cut
          elsif(/\s+SYSTEM STATS:\s+Time:([\d.]*)\s+.*Hosts:(\d+)\s+.*DataSources:(\d+)\s+.*RRDsProcessed:(\d+)/) {
            ($time,$hosts,$datasets,$rrds,$complete) = ($1,$2,$3,$4,1);
            print STDERR "SYSTEM STATS: time=$time, hosts=$hosts, datasources=$datasets, rrds=$rrds\n" if $DEBUG;
          }
=pod example of the failure line to be expected at the end:
2018/11/07 12:30:07 - PHPSVR ERROR: Input Expected, Script Server Terminating
=cut
          elsif(/\s+PHPSVR ERROR:.*Script Server Terminating/) {
            chomp;
            ($time,$complete,$error) = ($SPAN,1,$_);
            print STDERR "ERROR: $_" if $DEBUG;
          }
        }
        if (not $complete) {
            sleep $SLEEP;
            $log->clearerr();
        }
    }
    close $log;
    return {
        "time"      => [$time,    "n"],
        "hosts"     => [$hosts,   "i"],
        "rrds"      => [$rrds,    "i"],
        "items"     => [$items,   "i"],
        "datasets"  => [$datasets,"i"],
        "errors"    => [$errors,  "i"],
        "error"     => [$error,   "s"],
    };
};

1;
