package Site::Check_Mysql_Repl;

use strict;
use warnings;

use base 'Resmon::Module';

use Resmon::ExtComm qw(run_command);

=pod

=head1 Site::Check_Mysql_Repl

Site::Check_Mysql_Repl

=head1 SYNOPSIS

 Site::Check_Mysql_Repl {
     localhost : mysql => /path/to/mysql
     localhost : mycnf => /path/to/my.cnf
 }

=head1 DESCRIPTION

This module checks a slave mysql server running on localhost and detects 
any replication errors.

=head1 CONFIGURATION

=over

=item check_name

Check_Mysql_Repl

=item run

Command to run

=back

=head1 METRICS

=over

=item run

The command to run

=item lines

The number of lines in the output

=item words

The number of words in the output

=item chars

The length of the output

=item output

The output of the command

=back

=cut


sub handler {
    my $self = shift;
    my $config = $self->{config}; # All configuration is in here

    # Set some default values.
    my $CNF;
    my $MYSQL;
    my $user;
    my $pass;
    my $socket;
    my $error;

    if ( defined( $config->{mycnf} ) ) {
        $CNF = $config->{mycnf};
    }
    else {
        $CNF = "/root/.my.cnf";
    }

    if ( defined( $config->{mysql} ) ) {
        $MYSQL = $config->{mysql};
    }
    else {
        $MYSQL = "/usr/bin/mysql";
    }

    # Do some sanity checking.
    if ( ! -e "$MYSQL" ) {
        print STDERR "Could not find mysql binary\n";
        $error = 1;
    }

    if ( ! -e "$CNF" ) {
        print STDERR "Could not find my.cnf\n";
        $error = 2;
    }

    if ( defined( $error ) ) {
        my $threads_str = "Error $error";
        my $logfile_str = "Error $error";

        return {
            "threads"    => [$threads_str, "s"],
            "logfile"    => [$logfile_str, "s"],
        }
    }

    # Get the username and password from the my.cnf file.
    open( my $FH, "<", "$CNF" );

    foreach ( <$FH> ) {
        if ( $_ =~ m/^password\ ?=\ ?(.*)$/ ) {
            $pass = $1;
        }
        elsif ( $_ =~ m/^user\ ?=\ ?(.*)$/ ) {
            $user = $1;
        }
        elsif ( $_ =~ m/^socket\ ?=\ ?(.*)$/ ) {
            $socket = $1;
        }

        "a" =~ /a/;
    }

    close( $FH );

    # Now get the mysql slave status.
    my $mysql_args = '';
    if ( defined( $user ) ) {
        $mysql_args = $mysql_args . "-u $user";
    }
    if ( defined( $pass ) ) {
        $mysql_args = $mysql_args . " -p$pass";
    }
    if ( defined( $socket ) ) {
        $mysql_args = $mysql_args . " -S $socket";
    }

    my $result = run_command("$MYSQL $mysql_args -e 'show slave status;'");

    my @temparray = split( "\n", $result );

    my @fields = split( "\t", $temparray[0] );
    my @values = split( "\t", $temparray[1] );

    my $threads_str = '';
    my $logfile_str = '';

    # For certain fields, check the corresponding values.
    foreach ( 0..scalar( @fields )-1 ) {
        if ( $fields[$_] =~ m/^Slave_IO_Running$/ ) {
            if ( $values[$_] !~ m/^Yes$/i ) {
                $threads_str = "Slave IO Running: $values[$_]";
            }
        }
        elsif ( $fields[$_] =~ m/^Slave_SQL_Running$/ ) {
            if ( $values[$_] !~ m/^Yes$/i ) {
                if ( length( $threads_str ) > 0 ) {
                    $threads_str = $threads_str . ", ";
                }
                $threads_str = $threads_str . "Slave SQL Running: $values[$_]";
            }
        }
        elsif ( $fields[$_] =~ m/^Last_Errno$/ ) {
            if ( $values[$_] > 0 ) {
                $logfile_str = "Error number: $values[$_]";
            }
        }
        elsif ( $fields[$_] =~ m/^Last_Error$/ ) {
            if ( length( $values[$_] ) > 0 ) {
                if ( length( $logfile_str ) > 0 ) {
                    $logfile_str = $logfile_str . ", ";
                }
                $logfile_str = $logfile_str . "Error type: $values[$_]";
            }
        }
    }

    if ( length( $threads_str ) == 0 ) {
        $threads_str = 'OK';
    }
    if ( length( $logfile_str ) == 0 ) {
        $logfile_str = 'OK';
    }

    return {
        "threads"    => [$threads_str, "s"],
        "logfile"    => [$logfile_str, "s"],
    };
};

1;
