package Site::Command;

use strict;
use warnings;

use base 'Resmon::Module';

use Resmon::ExtComm qw(run_command cache_command);

=pod

=head1 Site::Command

Site::Command - a sample/template resmon module

=head1 SYNOPSIS

 Site::Command {
     some_check_name: run => command
 }

=head1 DESCRIPTION

This module runs the command and returns output of it along with numbers of chars, 
lines, and words in it

=head1 CONFIGURATION

=over

=item check_name

The check name is descriptive only in this check. It is not used for anything.
Some checks use the check_name as part of the configuration, such as
free space checks that specify the filesystem to use.

=item run

Command to run

=back

=head1 METRICS

=over

=item run

The command to run

=item lines

The number of lines in the output

=item words

The number of words in the output

=item chars

The length of the output

=item output

The output of the command

=back

=cut


sub handler {
    my $self = shift;
    my $config = $self->{config}; # All configuration is in here
    my $check_name = $self->{check_name}; # The check name is in here
    my $run = $config->{run}; # The "run" parameter

    my $output = cache_command($run, 600);
    my $chars  = length($output);
    my $words  = scalar(split(' ',$output));
    my $lines  = scalar(split(/^/,$output));
    $output =~ s/\n/;/mg;


    return {
        "run"    => [$run,    "s"],
        "output" => [$output, "s"],
        "chars"  => [$chars,  "i"],
        "words"  => [$words,  "i"],
        "lines"  => [$lines,  "i"]
    };
};

1;
