package Site::IMAP;

use strict;
use warnings;
use IO::Socket;
use Time::HiRes qw(tv_interval gettimeofday);

use base 'Resmon::Module';

use Resmon::ExtComm qw(run_command);

=pod

=head1 NAME

Site::IMAPlogin - check successful login and it's time

=head1 SYNOPSIS

 Site::IMAP {
    <username>: passwd => <passwd>
 }

 Site::IMAP {
    <username>: passwd => <passwd>, address => <IP address>
 }

=head1 DESCRIPTION

This module checks if login succeeds and what time it takes

=head1 CONFIGURATION

=over

=item check_name

The check name is username to use for logging into IMAP

=item passwd

Password to use for IMAP login

=back

=head1 METRICS

=over

=item connect

The time elapsed for getting IMAP banner

=item login

The time elapsed for login

=item list

The time elapsed for listing folders

=item select

The time elapsed for selecting INBOX

=item fetch

The time elapsed for fetching info

=item body

The time elapsed for fetching body

=item logout

The time elapsed for logout

=back

=cut

my $DEBUG=0;
sub handler {
  my $signature='hpjLESSlESTARLBRACEQYPLUSFxaRPARCARETSEMICOLONxwbLTILDATo';
  my $self = shift;
  my $config = $self->{config};
  my $user = $self->{check_name};
  my $pass = $config->{passwd};
  my ($connect,$login,$list,$select,$fetch,$body,$logout);
  my $t0=[gettimeofday];
  my $imap = IO::Socket::INET->new(
    Proto      => "tcp",
    PeerAddr   => "localhost",
    PeerPort   => "imap(143)",
  ) || die "can't connect to imap service on localhost: $!";
  $imap->autoflush(1);
  $_=<$imap>;
  if ($_) {
    if (/ Dovecot ready./) {
      $connect = tv_interval($t0);
      print $imap $signature."0 login $user $pass\r\n";
      until(/^$signature/) {$_=<$imap>};
      if (/ Logged in./) {
        $login = tv_interval($t0);
        print $imap $signature."1 list \"\" \"\*\"\r\n";
        $_=<$imap>;
        until(/^$signature/) {$_=<$imap>};
        if (/List completed./) {
          $list = tv_interval($t0);
          print $imap $signature."2 select \"INBOX\"\r\n";
          $_=<$imap>;
          until(/^$signature/) {$_=<$imap>};
          if (/Select completed./) {
            $select = tv_interval($t0);
            print $imap $signature."3 fetch 1 all\r\n";
            $_=<$imap>;
            until(/^$signature/) {$_=<$imap>};
            if (/Fetch completed./) {
              $fetch = tv_interval($t0);
              print $imap $signature."4 fetch 1 body[]\r\n";
              $_=<$imap>;
              until(/^$signature/) {$_=<$imap>};
              if (/Fetch completed./) {
                $body = tv_interval($t0);
                print $imap $signature."5 logout\r\n";
                $_=<$imap>;
                until(/^$signature/) {$_=<$imap>};
                if (/Logout completed./) {
                  $logout = tv_interval($t0);
                }
                else {
                  $logout=$_;
                }
              }
              else {
                $body=$_;
              }
            }
            else {
              $fetch=$_;
            }
          }
          else {
            $select=$_;
          }
        }
        else {
          $list=$_;
        }
      }
      else {
        $login=$_;
      }
    }
    else {
      $connect=$_;
    }
  };
  return { 
    "connect" => [$connect, "a"],
    "login" => [$login, "a"],
    "list" => [$list, "a"],
    "select" => [$select, "a"],
    "fetch" => [$fetch, "a"],
    "body" => [$body, "a"],
    "logout" => [$logout, "a"],
  };
};

1;
=pod for testing use:
my %h = ( 
   config => { passwd => 'yourpassword' },
   check_name => 'yourusername',
);
my $hr=\%h;
my $timing = handler($hr);
print STDERR 
    "connect =>  $timing->{connect}[0]\n",
    "login =>  $timing->{login}[0]\n",
    "list =>  $timing->{list}[0]\n",
    "select =>  $timing->{select}[0]\n",
    "fetch =>  $timing->{fetch}[0]\n",
    "body =>  $timing->{body}[0]\n",
    "logout =>  $timing->{logout}[0]\n"
;
=cut
