package Site::MailmanInLdap;

use strict;
use warnings;
use IO::Socket;
use Array::Diff;
use Time::HiRes qw(tv_interval gettimeofday);

use base 'Resmon::Module';

use Resmon::ExtComm qw(run_command);

=pod
=head1 NAME

Site::MailmanInLDap - check mailman lists and ldap aliases. 

=head1 SYNOPSIS

 Site::MailmanInLdap {
   lists: noop
 }

=head1 DESCRIPTION

This module checks if mailman lists for domain cs.umd.edu match correspondent mail aliases in LDAP

=head1 CONFIGURATION

=over

=item check_name

Arbitrary name of the check.

=head1 METRICS

=over

=item onlyinldap

=item onlyonmailman

=item count

=back
=cut

=pod
curl -q --user restadmin:restpass http://localhost:8001/3.1/domains/cs.umd.edu/lists | jq .entries[].fqdn_listname | sed -e 's/"//g'  | sort | tee /tmp/11
curl -q 'ldap://idm00.cs.umd.edu/cn=aliases,dc=cs,dc=umd,dc=edu?cn?sub?(mailHost=mailman3.cs.umd.edu)' 2>/dev/null | sed -n -e 's/^.*cn: //p' | sort | tee /tmp/00
=cut

sub handler {
    my $self = shift;
    my ($output_aliases,$output_mailman,$maillists,$aliases)=('','','','');
    my @suffixes=('bounces','confirm','join','leave','owner','request','subscribe','unsubscribe');
    $output_aliases = run_command("curl -q 'ldap://idm00.cs.umd.edu/cn=aliases,dc=cs,dc=umd,dc=edu?cn?sub?(mailHost=mailman3.cs.umd.edu)' 2>/dev/null | sed -n -e 's/^.*cn: //p' | sort");
    $output_mailman = run_command("curl -q --user restadmin:restpass http://localhost:8001/3.1/domains/cs.umd.edu/lists | jq .entries[].list_name | sed -e 's/\"//g' | sort" );
    my @aliases = split "\n", $output_aliases;
    my @saliases = sort @aliases;
    my @mlists = split "\n", $output_mailman;
    my @maliases = ();
    my $k=0;
    foreach my $list ( @mlists ) {
      push @maliases,"${list}\@cs.umd.edu";
      print "\nMA: =${list}\@cs.umd.edu=\n";
      print "\nLA: =$aliases[$k++]=\n";
      foreach my $suffix ( @suffixes ) {
        push @maliases,"${list}-${suffix}\@cs.umd.edu";
        print "\nMA: =${list}-${suffix}\@cs.umd.edu=\n";
        print "\nLA: =$aliases[$k++]=\n";
      }
    }
    my @msaliases = sort @maliases;
    my $diff = Array::Diff->diff( \@saliases, \@msaliases );
    my %metrics = (
        "onlyinldap"    => [ join(',',@{$diff->deleted}), "s"],
        "onlyonmailman" => [ join(',',@{$diff->added}),   "s"],
	"count"         => [ $diff->count, "i" ],
    );

    return \%metrics;
};

1;
