package Site::ROFS;

use strict;
use warnings;

use base 'Resmon::Module';

use Resmon::ExtComm qw(run_command);

=pod

=head1 NAME

Site::ROFS - Find the number of read-only filesystems

=head1 SYNOPSIS

 Site:ROFS {
     rofs: noop
 }

=head1 DESCRIPTION

This module queries /proc/mounts to find the number of filesystems mounted 
read-only.

=head1 CONFIGURATION

=over

=item check_name

The check name is descriptive only in this check. It is not used for anything.
Some checks use the check_name as part of the configuration, such as
free space checks that specify the filesystem to use.

=back

=head1 METRICS

=over

=item check_name

The name of the current check. You wouldn't normally return this, but it is
here to show how to access the check name, and for testing purposes.

=item count

Shows the number of filesystems which appear to be mounted read-only.

=back

=cut

sub handler {
    my $self = shift;
    my $config = $self->{config}; # All configuration is in here
    my $check_name = $self->{check_name}; # The check name is in here

    # Count the number of filesystems which appear to be mounted read-only.
    my $count = run_command("awk '\$4~/(^|,)ro(\$|,)/' /proc/mounts | wc -l");
    chomp $count;

    return {
        "check_name" => [$self->{check_name}, "s"],
        "count" => [$count, "i"],
        "command_result" => [$count, "s"]
    };
};

1;
