#! /usr/bin/env python

import os
import subprocess
import yaml
from argparse import ArgumentParser

parser = ArgumentParser()
parser.add_argument(
    '-f', '--file',
    dest='file',
    default='example.yml',
    help='YML file containing the testbed configuration'
   )

args = parser.parse_args()

with open(args.file) as cfile:
    conf = yaml.load(cfile)

ntwk = conf['network']
use_docker = ntwk.get('docker', False)
base_addr = reduce(
    lambda a,b : a*256+b,
    [ int(q) for q in ntwk['subnet_addr'].split('.') ]
   )
subnet = ntwk['subnet_addr']+'/'+str(ntwk['subnet_len'])

nodelist = list()

if use_docker:
    retval = subprocess.call(['docker','pull',conf['image']])

def make_ip(i):
    addr = base_addr + i
    masks = [ 24, 16, 8, 0 ]
    return '.'.join([ str((addr>>mask)&0xFF) for mask in masks ])
    
def make_route(f,net,fdev):
    subprocess.call([
        'sudo', 'ip', 'netns', 'exec', f['name'],
        'ip', 'route', 'add', net,
        'dev', fdev, 'proto', 'static', 'scope', 'global',
        'src', f['addr']
       ])

def connect_nodes(node1, node2, route_subnet=False):
    ctr1 = nodelist[node1]
    ctr2 = nodelist[node2]
    print ctr1, ctr2
    # Create a virtual ethernet link.
    dev_1 = '{name}_{n1}_{n2}'.format(name=ntwk['name'],n1=node1,n2=node2)
    dev_2 = '{name}_{n2}_{n1}'.format(name=ntwk['name'],n1=node1,n2=node2)
    subprocess.call([
        'sudo',
        'ip', 'link', 'add', dev_1, 'type', 'veth',
        'peer', 'name', dev_2
        ])

    # Add the link endpoints to the container namespaces.
    def add_to_ns(dev,ns):
        subprocess.call([
            'sudo', 'ip', 'link', 'set', dev, 'netns', ns
           ])
    add_to_ns(dev_1,ctr1['name'])
    add_to_ns(dev_2,ctr2['name'])

    # Set the IP addresses.
    def set_addr(dev,ns,addr):
        subprocess.call([
            'sudo', 'ip', 'netns', 'exec', ns,
            'ip', 'addr', 'add', addr+'/32', 'dev', dev
           ])
    set_addr(dev_1,ctr1['name'],ctr1['addr'])
    set_addr(dev_2,ctr2['name'],ctr2['addr'])

    # Bring the link up.
    def link_up(ns,dev):
        subprocess.call([
            'sudo', 'ip', 'netns', 'exec', ns,
            'ip', 'link', 'set', dev, 'up'
           ])
    link_up(ctr1['name'],dev_1)
    link_up(ctr2['name'],dev_2)

    # Set initial routes.
    def route_from_to(f,t,fdev):
        make_route(f,t['addr']+'/32',fdev)
    route_from_to(ctr1,ctr2,dev_1)
    route_from_to(ctr2,ctr1,dev_2)
    if route_subnet:
        make_route(ctr2,subnet,dev_2)

    # Configure the link for network emulation.
    def enable_netem(ns, dev):
	print "NAMESPACE: %s | DEVICE %s" % (ns,dev)
        subprocess.call([
            'sudo', 'ip', 'netns', 'exec', ns,
            'tc', 'qdisc', 'add', 'dev', dev, 'root', 'handle', '1:0', 'netem'
            ])
    enable_netem(ctr1['name'],dev_1)
    enable_netem(ctr2['name'],dev_2)
    return (dev_1,dev_2)


def make_mesh():
    global nodelist
    num_nodes = len(nodelist)
    [ connect_nodes(n1,n2)
        for n1 in range(num_nodes-1)
        for n2 in range(n1+1,num_nodes) ]

def make_star():
    global nodelist
    def get_dev(i):
        return '{name}_{i}_0'.format(name=ntwk['name'],i=i)
    bridge_ns = nodelist[0]['name']
    bridge_dev = '{}_bridge'.format(bridge_ns)
    bridge_addr = nodelist[0]['addr']
    subprocess.call([
        'sudo', 'ip', 'netns', 'exec', bridge_ns,
        'brctl', 'addbr', bridge_dev
    ])
    subprocess.call([
        'sudo', 'ip', 'netns', 'exec', bridge_ns,
        'ip', 'address', 'add', 'dev', bridge_dev, 'local', bridge_addr
    ])
    subprocess.call([
        'sudo', 'ip', 'netns', 'exec', bridge_ns,
        'ip', 'link', 'set', 'dev', bridge_dev, 'up'
    ])

    num_nodes = len(nodelist)
    devs = [ connect_nodes(0,n,True)[0] for n in range(1,num_nodes) ]
    for d in devs:
        subprocess.call([
            'sudo', 'ip', 'netns', 'exec', bridge_ns,
            'brctl', 'addif', bridge_dev, d
        ])

def start_node(base_name,i):
    global nodelist
    name = base_name + str(i)

    if use_docker:
        # Start the container.
        command = ['docker',
                   'run',
                   '-d',
                   '--name',
                   name
                  ]
        if conf['docker_opts']:
            command.extend(conf['docker_opts'].split())
        command.append(conf['image'])
        command.extend(conf['image_cmd'].split())
        retval = subprocess.call(command)

    entry = dict()
    entry['name'] = name
    # Determine the IP address to use for this node.
    entry['addr'] = make_ip(i)
    if use_docker:
        # Find the container's process ID.
        entry['pid'] = subprocess.check_output([
            'docker',
            'inspect',
            '-f',
            '{{.State.Pid}}',
            name
            ]).strip()
 
        # Set up a network namespace.
        subprocess.call([
            'sudo',
            'ln',
            '-s',
            '/proc/{pid}/ns/net'.format(pid=entry['pid']),
            '/var/run/netns/{name}'.format(name=name)
           ])
    else:
        # Create the network namespace for the node, without docker.
        subprocess.call([
            'sudo',
            'ip',
            'netns',
            'add',
            name
           ])
        subprocess.call([
            'sudo',
            'ip',
            'netns',
            'exec',
            name,
            'ip',
            'link',
            'set',
            'lo',
            'up'
           ])
        pass

    # Add the node's info to our list.
    nodelist.append(entry)

# First, make sure that /var/run/netns exists.
subprocess.call([ 'sudo', 'mkdir', '-p', '/var/run/netns' ])

for n in range(ntwk['num_nodes']):
    start_node(ntwk['name'],n)

if ntwk['topology'] == 'mesh':
    make_mesh()
else:
    make_star()

